
/**
 * \file 
 * \brief   Dynamically allocated string struct.
 * \author  Felix Ott
 */

#ifndef ASTR_H__
#define ASTR_H__

#include <stdlib.h>

/** \brief  Resizable string struct. */
typedef struct astr astr;
/** \brief  Function type for determining the new capacity after resize. */
typedef size_t(*astr_capacityfunc_t)(size_t current_capacity,size_t newsize);

/**
 * \brief   Allocate an \ref astr struct.
 * \param   str Initial contents, may be NULL.
 * \return  Pointer to the created astr, or NULL on malloc failure.
 */
astr *astr_create(const char *const str);

/** \brief  Deallocate an \ref astr struct. */
void astr_destroy(astr *const s);

/** \brief  Pointer to the start of the stored string. */
char *astr_beg(astr *const s);
/** \brief  Pointer to the end of the stored string. */
char *astr_end(astr *const s);
/** \brief  Get length of the stored string. */
size_t astr_len(const astr *const s);

/**
 * \brief   Assign contents of one \ref astr to another.
 * \return  0: success, -1: allocation failure
 */
int astr_ass(astr *const s,const astr *t);

/**
 * \brief  Assign a character string to an \ref astr.
 * \return  0: success, -1: allocation failure
 */
int astr_assc(astr *const s,const char *const str);

/** 
 * \brief  Assign part of a character string to an \ref astr.
 * \return  0: success, -1: allocation failure
 */
int astr_asscn(astr *const s,const char *const str,size_t n);

/**
 * \brief  Append contents of one \ref astr to another.
 * \return  0: success, -1: allocation failure
 */
int astr_app(astr *const s,const astr *t);

/**
 * \brief  Append a character string to an \ref astr.
 * \return  0: success, -1: allocation failure
 */
int astr_appc(astr *const s,const char *const str);

/**
 * \brief  Append part of a character string to an \ref astr.
 * \return  0: success, -1: allocation failure
 */
int astr_appcn(astr *const s,const char *const str,size_t n);

/**
 * \brief  Resize an \ref astr.
 * \return  0: success, -1: allocation failure
 */
int astr_resize(astr *const s,size_t size);

/** 
 * \brief  Print a value to an \ref astr at offset ofs (cf. sprintf()).
 * \return  0: success, -1: allocation failure
 */
int astr_printf(astr *const s,size_t ofs,const char *fmt,...);

/** \brief  Set a function for determining the new capacity after resize. */
void astr_capacityfunc(astr *const s,astr_capacityfunc_t func);

/**
 * \brief   Function for determining the new capacity after resize.
 *  
 * With this function, astr_resize() always changes the capacity to fit the
 * contents optimally.
 */
size_t astr_capacityfunc_fit(size_t current_capacity,size_t newsize);

/**
 * \brief   Function for determining the new capacity after resize.
 *  
 * With this function, astr_resize() changes the capacity to fit the contents
 * optimally if new memory must be allocated. But The capacity won't shrink
 * when resizing to a smaller size.
 */
size_t astr_capacityfunc_noshrink(size_t current_capacity,size_t newsize);

/**
 * \brief   Function for determining the new capacity after resize.
 *  
 * With this function, astr_resize() will grow the capacity exponentially
 * if new memory must be allocated.
 */
size_t astr_capacityfunc_exp(size_t current_capacity,size_t newsize);

#endif /* ASTR_H__ */

