
/**
 * \file 
 * \brief   Dynamically allocated string struct.
 * \author  Felix Ott
 */

#include "astr.h"

#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

struct astr
{
    size_t size;
    size_t capacity;
    astr_capacityfunc_t capacity_func;
    char *data;
};

astr *astr_create(const char *const str)
{
    astr *ret=malloc(sizeof(astr));

    if(ret==0) return 0;

    ret->size=(str==0)?0:strlen(str);
    ret->capacity=ret->size+1;
    ret->data=malloc(ret->capacity);
    ret->capacity_func=astr_capacityfunc_fit;

    if(ret->data==0)
    {
        free(ret);
        return 0;
    }

    if(str!=0) strcpy(ret->data,str);
    return ret;
}

void astr_destroy(astr *const s)
{
    if(s!=0)
    {
        free(s->data);
        free(s);
    }
}

char *astr_beg(astr *const s)
{
    return s->data;
}

char *astr_end(astr *const s)
{
    return s->data+s->size;
}

size_t astr_len(const astr *const s)
{
    return s->size;
}

int astr_ass(astr *const s,const astr *t)
{
    if(astr_resize(s,t->size)!=0) return -1;
    strcpy(s->data,t->data);
    return 0;
}

int astr_assc(astr *const s,const char *const str)
{
    if(str==NULL)
    {
        if(astr_resize(s,0)) return -1;
    }
    else
    {
        if(astr_resize(s,strlen(str))!=0) return -1;
        strcpy(s->data,str);
    }
    return 0;
}

int astr_asscn(astr *const s,const char *const str,size_t n)
{
    if(str==NULL)
    {
        if(astr_resize(s,0)) return -1;
    }
    else
    {
        size_t len=0;
        while((n>0)&&(str[len]!=0))
        {
            ++len;
            --n;
        }

        if(astr_resize(s,len)!=0) return -1;
        memcpy(s->data,str,len);
    }
    return 0;
}

int astr_app(astr *const s,const astr *t)
{
    const size_t oldsize=s->size;
    if(astr_resize(s,s->size+t->size)!=0) return -1;
    strcpy(s->data+oldsize,t->data);
    return 0;
}

int astr_appc(astr *const s,const char *const str)
{
    if(str==NULL) return 0;
    const size_t oldsize=s->size;
    if(astr_resize(s,s->size+strlen(str))!=0) return -1;
    strcpy(s->data+oldsize,str);
    return 0;
}

int astr_appcn(astr *const s,const char *const str,size_t n)
{
    if(str==NULL) return 0;
    size_t len=0;
    while((n>0)&&(str[len]!=0))
    {
        ++len;
        --n;
    }

    const size_t oldsize=s->size;
    if(astr_resize(s,s->size+len)!=0) return -1;
    memcpy(s->data+oldsize,str,n);
    return 0;
}

int astr_resize(astr *const s,size_t size)
{
    const size_t capac=s->capacity_func(s->capacity,size);

    if(capac!=s->capacity)
    {
        char* d=realloc(s->data,capac);    
        if(d==0) return -1;
        s->data=d;
        s->capacity=capac;
    }

    s->data[size]=0;
    s->size=size;

    return 0;
}

int astr_printf(astr *const s,size_t ofs,const char *fmt,...)
{
    int rc;
    if(s->size<ofs)
    {
        rc=astr_resize(s,ofs);
        if(rc<0) return -1;
    }
    size_t limit=s->size-ofs+1;
    size_t n;

    va_list ap;
    va_start(ap,fmt);
    n=vsnprintf(s->data+ofs,limit,fmt,ap);
    va_end(ap);

    if(n>=limit)
    {
        rc=astr_resize(s,ofs+n);
        if(rc<0) return -1;

        limit=n+1;
        va_start(ap,fmt);
        n=vsnprintf(s->data+ofs,limit,fmt,ap);
        va_end(ap);

        if(n>=limit) return -1;
    }

    return n;
}

size_t astr_capacityfunc_fit(size_t current_capacity,size_t newsize)
{
    return newsize+1;
}

size_t astr_capacityfunc_noshrink(size_t current_capacity,size_t newsize)
{
    return (newsize<current_capacity)?current_capacity:(newsize+1);
}

size_t astr_capacityfunc_exp(size_t current_capacity,size_t newsize)
{
    while(newsize>=current_capacity)
    {
        current_capacity=current_capacity<<1;
    }
    return current_capacity;
}

void astr_capacityfunc(astr *const s,astr_capacityfunc_t func)
{
    s->capacity_func=func;
}

