// Authors: Korbinian Schneeberger, Joerg Hagman and Sandra Gesing
// Copyright (C) 2008 by Max-Planck Institute for Developmental Biology, Tuebingen, Germany

#include "genomemapper.h"
#include <sys/time.h>

int size_hit(HIT *hit, unsigned int *oldlength);
int duplicate(HIT* hit);
void printgenome();
#ifdef THREADS
void *prepare_align(void *arg);
#endif 
#ifdef METHYLOME
int features_conversion(HIT* hit);
#endif

int map_reads() {

	char eof = 0;
	char read_mapped;
	unsigned int count_reads = 0;
	int first_slot = 0, nr_seeds;
	int num_edit_ops = NUM_EDIT_OPS;
	
	//FILE *CHROM_ENTRY_FP;
	//CHROM_ENTRY_FP = fopen("MappingEntriesPerRead", "w");

#ifdef THREADS
	long j;
	int i;
	NEW_READ = 1;
        pthread_attr_init(&attr);
        pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_JOINABLE);
        pthread_attr_setscope (&attr, PTHREAD_SCOPE_SYSTEM);

        pthread_mutex_init(&num_edit_ops_mutex, NULL);
        pthread_mutex_init(&hits_into_score_mutex, NULL);
	for (j = 0; j < NUM_THREADS; j++) {
		pthread_mutex_init(&start_align_mutex[j], NULL);
		pthread_cond_init(&start_align_cond[j], NULL);
	}
	pthread_mutex_init(&aligned_mutex, NULL);
	pthread_cond_init(&aligned_cond, NULL);
        tid = (pthread_t*) malloc ((NUM_THREADS) * sizeof(pthread_t));

	for (j = 0; j < NUM_THREADS; j++) {
                if (pthread_create(&tid[j], &attr, prepare_align, (void*) j))
                        fprintf(stderr, "ERROR: THREAD %ld could not be created.\n", j);
	}
	
#endif

	while (!eof) {
		
		count_reads++;
		if (DEBUGREAD == count_reads) DEBUG = 1; else DEBUG = 0;

#ifdef THREADS
		NUM_HITS = 0;
		for (i = 0; i < MAX_READ_LENGTH; i++)
                	HITS_NUM_PER_LENGTH[i] = 0;
#endif
		
		eof = read_short_read();
		if (eof) continue;

		if (READ_LENGTH < HITLEN_LIMIT) {
			fprintf(stderr, "\n!!! WARNING! Read %d (%s) with length %d is shorter than the hitlength limit (=%d) and will not be processed!\n\n",
			count_reads, READ_ID, READ_LENGTH, HITLEN_LIMIT);
		}
		else {
//			printf("%d ",count_reads); fflush(stdout);

			// progress output, just for user convenience
			if (VERBOSE && (count_reads % 10000 == 0)) {
				printf(".");
				fflush(stdout);
			}
			
			LONGEST_HIT = 0;
			HITS_IN_SCORE_LIST = 0;

			BEST_SCORE = WORST_SCORE + 1;
			SCORE_LIMIT = WORST_SCORE + 1;

			///////////////////////////////////////////////////////////////////////////////////////
			// map_fast IF 1) best hit strategy
			//             2) only hits up to READLENGTH/IDEX_DEPTH mismatches without gaps should be found
			nr_seeds = (int) (READ_LENGTH / INDEX_DEPTH);
 			if ( (GAP_SCORE > MM_SCORE) && ( ALL_HIT_STRATEGY == 0 || (NUM_MISMATCHES < nr_seeds && NUM_GAPS == 0) ) ) {
 				first_slot = map_fast();
 			}

 			// map_complete IF 1) all hit strategy 2) best hit strategy and no mappings found in map_fast BUT NOT IF MM < RL/ID AND gaps=0 (since map_fast has already found them)
			//if (ALL_HIT_STRATEGY != 0 && !(NUM_MISMATCHES < nr_seeds && NUM_GAPS == 0)) {
			if ((ALL_HIT_STRATEGY != 0 && !(NUM_MISMATCHES < nr_seeds && NUM_GAPS == 0)) || GAP_SCORE <= MM_SCORE) {
#ifndef METHYLOME	
				map_short_read(count_reads, first_slot);
				dealloc_mapping_entries();
				CHROMOSOME_ENTRY_OPERATOR->used = 0;
#else
				map_short_read(count_reads, first_slot, 1);
                                dealloc_mapping_entries();
                                CHROMOSOME_ENTRY_OPERATOR->used = 0;

				map_short_read(count_reads, first_slot, 2);
                                dealloc_mapping_entries();
                                CHROMOSOME_ENTRY_OPERATOR->used = 0;
#endif
				browse_hits();
			}

			if (ALL_HIT_STRATEGY < 0) ALL_HIT_STRATEGY = 0;
			
			NUM_EDIT_OPS = num_edit_ops;	// resetting NUM_EDIT_OPS - has been changed probably in alignments and map_fast

			/*NUM_SCORE_INTERVALS = (int) SCORE_LIMIT / SCORE_INTERVAL;
			if (NUM_SCORE_INTERVALS * SCORE_INTERVAL != (int) SCORE_LIMIT) ++NUM_SCORE_INTERVALS;
			NUM_SCORE_INTERVALS++;
printf("hit num_score_intervals: %d, scorelimit %d bestscore %d\n",NUM_SCORE_INTERVALS,SCORE_LIMIT, BEST_SCORE);*/

			read_mapped = print_hits();	// returns 1 if at least one hit is printed, 0 otherwise
			
			if (read_mapped) READS_MAPPED++;
			else {
				int printed = 0;
				if (SPLICED_HITS) {
					printed = print_spliced_hits();	
					if (printed == 0) {
						printed = print_largest_hit();
					}
				}
				if (strlen(LEFTOVER_FILE_NAME) > 0) print_leftovers();
			}
			
			dealloc_mapping_entries(); //muss eigentlich nur der container zaehler zurückgesetzt werden... optimization?
			dealloc_hits();
			dealloc_hits_by_score();
			CHROMOSOME_ENTRY_OPERATOR->used = 0;
			if (LONGEST_HIT != 0) dealloc_hit_lists_operator();
		}

	}
#ifdef THREADS
	NEW_READ = 0;
        for (j = 0; j < NUM_THREADS; j++) {
		while (pthread_mutex_trylock(&start_align_mutex[j])) usleep(1);
		pthread_cond_signal(&start_align_cond[j]);
		pthread_mutex_unlock(&start_align_mutex[j]);
	}

        pthread_attr_destroy(&attr);
        for (j = 0; j < NUM_THREADS; j++)
                pthread_join(tid[j], NULL);
        pthread_mutex_destroy(&num_edit_ops_mutex);
        pthread_mutex_destroy(&hits_into_score_mutex);
        for (j = 0; j < NUM_THREADS; j++) {
		pthread_mutex_destroy(&start_align_mutex[j]);
		pthread_cond_destroy(&start_align_cond[j]);
	}
	pthread_mutex_destroy(&aligned_mutex);
	pthread_cond_destroy(&aligned_cond);

	free(tid);
	free(THREAD_STARTED);
	free(start_align_mutex);
	free(start_align_cond);
        for (j=0; j!=NUM_THREADS; ++j) {
                for (i=0; i!=2*NUM_GAPS+1; ++i) {
                        free(M[j][i]);
                        free(T[j][i]);
                }
		free(M[j]);
		free(T[j]);
        }
	free(M);
	free(T);

	free(HITS_NUM_PER_LENGTH);

#endif
	
	if (strlen(OUT_FILE_NAME) > 0) fclose(OUT_FP);
	if (strlen(LEFTOVER_FILE_NAME) > 0) fclose(LEFTOVER_FP);
	
	NUM_READS = count_reads - 1;
	
	if (VERBOSE) printf("\n");
	
	return(0);
}

#ifndef METHYLOME
int seed2genome(unsigned int num, unsigned int index_slot, unsigned int readpos, char reverse) {
#else
int seed2genome(unsigned int num, unsigned int index_slot, unsigned int readpos, char reverse, int conversion) {

	POS *BLOCK_TABLE;
	if (conversion == 1) {
		BLOCK_TABLE = BLOCK_TABLE_CT;
	}
	else {
		BLOCK_TABLE = BLOCK_TABLE_GA;
	}
#endif

	INDEX_ENTRY index_entry;
	STORAGE_ENTRY *index_mmap;
	unsigned int genome_pos, genome_chr, block;
	unsigned char pos;
	char strand;
	unsigned int direction;
	unsigned int mmoffset; //Mismatch-Offset


	char flag = 0;

	CHROMOSOME_ENTRY *chromosome_director, *chromosome_director_neighbor, *chromosome_director_new;
	MAPPING_ENTRY *mapping_entry, *existing_entry, *neighbor;

	HIT *hit = NULL;

	unsigned int i;
	unsigned int oldlength;
	
	// reverse = 1: only index, 2: only index_rev, 4: both
	while (reverse > 0) {
		
		if (reverse != 2) {
#ifndef METHYLOME
			index_mmap = INDEX_FWD_MMAP;
			index_entry = *(INDEX+index_slot);
			direction = -1;
#else
			if (conversion == 1) {
				index_mmap = INDEX_FWD_CT_MMAP;
        	                index_entry = *(INDEX_CT+index_slot);
                	        direction = -1;
			}
			else {
				index_mmap = INDEX_FWD_GA_MMAP;
                                index_entry = *(INDEX_GA+index_slot);
                                direction = -1;
			}
#endif
		}
		else {
#ifndef METHYLOME
			index_mmap = INDEX_REV_MMAP;
			index_entry = *(INDEX_REV+index_slot);
			direction = 1;
#else
			if (conversion == 1) {
				index_mmap = INDEX_REV_CT_MMAP;
        	                index_entry = *(INDEX_REV_CT+index_slot);
                	        direction = 1;
			}
			else {
				index_mmap = INDEX_REV_GA_MMAP;
                                index_entry = *(INDEX_REV_GA+index_slot);
                                direction = 1;
			}
#endif
		}

		for (i=0; i<index_entry.num; i++) { // foreach seed...
		
			// Every mapping gets an entry
			mapping_entry = alloc_mapping_entry();
			mapping_entry->readpos = readpos;

			// Get current position in the chromosome
			if (reverse != 2) {
				block = 0;
				memcpy(&block, &((index_mmap+(index_entry.offset-(i+1)))->id[0]), 3 * sizeof(char));
				pos = 0;
				memcpy(&pos, &((index_mmap+(index_entry.offset-(i+1)))->id[3]), sizeof(unsigned char));	
				genome_pos = pos + BLOCK_TABLE[block].pos;
				genome_chr = BLOCK_TABLE[block].chr;
				strand = 1;
			}
			else {
				block = 0;
				memcpy(&block, &((index_mmap+(index_entry.offset-(index_entry.num-i)))->id[0]), 3 * sizeof(char));
				pos = 0;
				memcpy(&pos, &((index_mmap+(index_entry.offset-(index_entry.num-i)))->id[3]), sizeof(unsigned char));
				genome_pos = pos + BLOCK_TABLE[block].pos;
				genome_chr = BLOCK_TABLE[block].chr;
				strand = -1;
			}
			
	
			// Check if there is already a chromosome director and get this or a new one
			if (*(GENOME+genome_pos) == NULL) {
                		flag = 1;


		                chromosome_director = alloc_chromosome_entry(&genome_pos, &genome_chr, &strand);
                		if (!chromosome_director) return(0);    // chrom_container_size too small -> cancel this read

				*(GENOME + genome_pos) = chromosome_director;

			}
		        else {

                	       chromosome_director = *(GENOME + genome_pos);
 			
				// is chrom_director from the actual read or from a former one?
      				if (chromosome_director >= (CHROMOSOME_ENTRY_OPERATOR->entries+CHROMOSOME_ENTRY_OPERATOR->used) 
					|| (chromosome_director < (CHROMOSOME_ENTRY_OPERATOR->entries+CHROMOSOME_ENTRY_OPERATOR->used) && 
					(chromosome_director->genome_pos != genome_pos))) {

                		    // it's from a former read, thus we need a new chrom_director:
                    		    chromosome_director = alloc_chromosome_entry(&genome_pos, &genome_chr, &strand);
			       	    if (!chromosome_director) return(0);    // chrom_container_size too small -> cancel this read

					*(GENOME + genome_pos) = chromosome_director;

				}
			}

			// Parse the list of chromosome directors
			while (chromosome_director->next != NULL && (chromosome_director->chromosome != genome_chr || (chromosome_director->chromosome == genome_chr && chromosome_director->strand != strand)))
			{
				chromosome_director = chromosome_director->next;
			}

			// Chromosome director is set, but still it could be the wrong chromosome, if the right chromosome is not in there so far.
			if (chromosome_director->chromosome != genome_chr || chromosome_director->strand != strand) {
				chromosome_director_new = alloc_chromosome_entry(&genome_pos, &genome_chr, &strand);
				if (!chromosome_director_new) return(0);
				
				chromosome_director_new->next = chromosome_director->next;
				chromosome_director->next = chromosome_director_new;

				chromosome_director = chromosome_director_new;
			}
			
			// Paste MAPPING_ENTRY in list of chromosome director slot
			if (chromosome_director->mapping_entries == NULL) {
				chromosome_director->mapping_entries = mapping_entry;
			}
			else {
				if (flag == 1) {
					printf("!!!!!!!!!!!Found entry in chr dir at genome pos %d  --  %p\n", genome_pos, chromosome_director->mapping_entries	);
					exit(1);
				}

				existing_entry = chromosome_director->mapping_entries;
				mapping_entry->pred = existing_entry;
				existing_entry->succ = mapping_entry;
				chromosome_director->mapping_entries = mapping_entry;
			}


			// HIT EXTENSION
			
			//Check left (for plus strand) and right (for minus strand) neighbor at the genomeposition of the hit if there is a hit to join.
			if (genome_pos > 0 && readpos > 1) {
				
				if (*(GENOME + (genome_pos + direction)) != NULL) { // Is there a chromosome director?
					chromosome_director_neighbor = *(GENOME + (genome_pos + direction));
					
					// Is the chrom director from actual read?
					if (chromosome_director_neighbor < (CHROMOSOME_ENTRY_OPERATOR->entries+CHROMOSOME_ENTRY_OPERATOR->used) &&
					   (chromosome_director_neighbor->genome_pos == genome_pos + direction)) {
			
						// is there a mapping entry in the right chromosome solt, mr. director?
						// Search for chromosome_director_neighbor for the correct chromosome and strand
						while (chromosome_director_neighbor->next != NULL && (chromosome_director_neighbor->chromosome != genome_chr || 
						      (chromosome_director_neighbor->chromosome == genome_chr && chromosome_director_neighbor->strand != strand)))
						{
							chromosome_director_neighbor = chromosome_director_neighbor->next;
						}

						if (chromosome_director_neighbor->chromosome == genome_chr && chromosome_director_neighbor->strand == strand)
						{
							neighbor = chromosome_director_neighbor->mapping_entries;
							if (neighbor->readpos == mapping_entry->readpos-1) { // is the neighbored mapping entry also neighbored in the read?
								hit = neighbor->hit;
								if (hit != NULL) {
									oldlength = hit->end - hit->start + 1;
									mapping_entry->hit = neighbor->hit;
									if (reverse != 2) hit->end++;
									else hit->start--;
								}
							}
						}
					}
				}
				
				

				// MISMATCH extension 
				
				//combine with possible hit at position seedlength+1 to the left(+) or right(-) to span hit over mismatch
				if (hit == NULL && NUM_MISMATCHES != 0) {
					mmoffset = (reverse != 2)? -INDEX_DEPTH - 1: INDEX_DEPTH + 1;
					
					if ( (genome_pos + mmoffset > 0) && (genome_pos + mmoffset < CHR_LENGTH[genome_chr]) && (*(GENOME + (genome_pos + mmoffset)) != NULL) ) {
						chromosome_director_neighbor = *(GENOME + (genome_pos + mmoffset));
						
						// Is the chrom director from actual read?
						if  (chromosome_director_neighbor < (CHROMOSOME_ENTRY_OPERATOR->entries+CHROMOSOME_ENTRY_OPERATOR->used) && (chromosome_director_neighbor->genome_pos == genome_pos + mmoffset)) {
							
							// is there a mapping entry in the right chromosome solt, mr. director?
							// Search for chromosome_director for the correct chromosome
							while (chromosome_director_neighbor->next != NULL && (chromosome_director_neighbor->chromosome != genome_chr || 
							      (chromosome_director_neighbor->chromosome == genome_chr && chromosome_director_neighbor->strand != strand)))
							{
								chromosome_director_neighbor = chromosome_director_neighbor->next;
							}
	
							if (chromosome_director_neighbor->chromosome == genome_chr && chromosome_director_neighbor->strand == strand) {
								neighbor = chromosome_director_neighbor->mapping_entries;
								if (neighbor->readpos == mapping_entry->readpos - INDEX_DEPTH - 1) {
									
									//if ((neighbor->hit)->mismatches < NUM_EDIT_OPS && (neighbor->hit)->mismatches-(neighbor->hit)->gaps < NUM_MISMATCHES) {
									if ((neighbor->hit)->mismatches < NUM_MISMATCHES) {
										hit = neighbor->hit;
										oldlength = hit->end - hit->start + 1;
										mapping_entry->hit = neighbor->hit;
										
										if (reverse != 2) {
											hit->end = hit->end + INDEX_DEPTH + 1;
											hit->edit_op[hit->mismatches].pos = readpos - 1;
											hit->edit_op[hit->mismatches].mm = 1;
										}
										else {
											hit->start = genome_pos+1;
											hit->edit_op[hit->mismatches].pos = READ_LENGTH - readpos + 2;
											hit->edit_op[hit->mismatches].mm = 1;
										}
										hit->mismatches++;
									}
								}
							}
						}
					}
				}
			}
	
			// for MM=0: if potential hit doesn't start at readpos 1, it cannot become perfect, thus it is not even allocated:
			if ( !(NUM_MISMATCHES == 0 && readpos != 1) ) {

				// create new hit:
				if (hit == NULL) {
					
					hit = alloc_hit();
					mapping_entry->hit = hit;
			
					hit->chromosome = genome_chr;
					hit->end = genome_pos + INDEX_DEPTH;
					hit->orientation = (reverse != 2)? '+': '-';
					
					oldlength = INDEX_DEPTH - 1;
						
					hit->readpos = readpos;
					hit->start = genome_pos+1;

#ifdef METHYLOME
					hit->conversion = conversion;
#endif
				}
				
			}
	
			// for MM=0: if hit doesn't start at readpos=1, then do not report hit (due to: all perfect hits are extended from a hit starting at readpos 1!!!)
			if ( hit != NULL && !(NUM_MISMATCHES == 0 && hit->readpos != 1) ) {
				
				size_hit(hit, &oldlength);
				
			}
			
			//printgenome();	
			
			flag = 0;
			hit = NULL;
			
		} //end of for each seed on read

		reverse -= 2; // 1->-1, 2->0, 4->2


	} //end of while (for each strand)

	
	return(1);

}

void printgenome()
{
	printf("G E N O M E: \n");
	unsigned int i,c;
	HIT *hit;
	CHROMOSOME_ENTRY *ce;
	for (i=0; i!=LONGEST_CHROMOSOME; ++i) {
		c=0;
		ce = *(GENOME+i);
		if (ce != NULL) {
			printf("%d: ",i);
			while (ce != NULL) {
				++c;
				printf("(%d, %d, %d, %d) ", ce->chromosome+1, ce->strand, ce->genome_pos, (ce->next!=NULL));
				hit = ce->mapping_entries->hit;
				printhit(hit);
				ce = ce->next;
			}
			printf("((%d))\n",c);
		}
	}	
}

int size_hit(HIT *hit, unsigned int *oldlength)
{
	HIT *last, *next;

	// close the gap where the hit is taken out
	// shortest possible hits are not under control of the operator so far
	if (*oldlength > INDEX_DEPTH - 1) {
		
		if (hit->last == NULL) { //hit is the first in the list, the operator must be re-linked
			*(HIT_LISTS_OPERATOR+*oldlength) = hit->next;
			next = hit->next;
			if (next != NULL) {
				next->last = NULL;
			}
		}
		else {
			//sweet
			last = hit->last;
			next = hit->next;
			last->next = next;
			if (next != NULL) {
				next->last = last; 
			}
		}
#ifdef THREADS
		NUM_HITS--;
		HITS_NUM_PER_LENGTH[*oldlength]--;
#endif
	}
	unsigned int length = hit->end - hit->start + 1;
	
	// ##########################################################
	// Mismatch at first or last pos of read:
	// ##########################################################
	
	// if hit.readpos == 2, then first base of read must be mismatch (spares alignment):
	if (hit->readpos == 2) {
		//if (hit->mismatches < NUM_EDIT_OPS && hit->mismatches-hit->gaps < NUM_MISMATCHES) {
		if (hit->mismatches < NUM_MISMATCHES) {
			if (hit->orientation == '+' && hit->start != 1) {
					hit->start--;
					hit->edit_op[hit->mismatches].pos = 1;
					hit->edit_op[hit->mismatches].mm = 1;
					hit->mismatches++;
					hit->readpos--;
					// update length:
					length++;
			}
			else if (hit->orientation == '-' && hit->end != CHR_LENGTH[hit->chromosome]) {
					hit->end++;
					hit->edit_op[hit->mismatches].pos = READ_LENGTH;
					hit->edit_op[hit->mismatches].mm = 1; 
					hit->mismatches++;
					hit->readpos--;
					// update length:
					length++;
			}
		}
		else {	// do not add hit to hit list
			return 0;
		}
	}
	
	// if hit ends at pos |read|-1, then check last base of read if it's a mismatch:
	if (hit->readpos + length == READ_LENGTH) {

		// compare last bases:
		READ[READ_LENGTH - 1] = toupper(READ[READ_LENGTH - 1]);
		if ( ( 
				(hit->orientation == '+') && (hit->end != CHR_LENGTH[hit->chromosome]) && ( (READ[READ_LENGTH - 1] != CHR_SEQ[hit->chromosome][hit->end]) || !unique_base(READ[READ_LENGTH - 1]) ) 
			 ) || ( 
		   	 	(hit->orientation == '-') && (hit->start != 1) && ( (READ[READ_LENGTH - 1] != get_compl_base(CHR_SEQ[hit->chromosome][hit->start-2])) || !unique_base(READ[READ_LENGTH - 1]) )
		   	 ) ) 
		{
			
			// create mismatch:
			//if (hit->mismatches < NUM_EDIT_OPS && hit->mismatches-hit->gaps < NUM_MISMATCHES) {
			if (hit->mismatches < NUM_MISMATCHES) {
				//if (hit->orientation == '+' && hit->end != CHR_LENGTH[hit->chromosome]) {
				if (hit->orientation == '+') {
						hit->end++;
						hit->edit_op[hit->mismatches].pos = READ_LENGTH;
						hit->edit_op[hit->mismatches].mm = 1; 
						hit->mismatches++;
						// update length:
						length++;
				}
				//else if (hit->orientation == '-' && hit->start != 0) {
				else if (hit->orientation == '-') {
						hit->start--;
						hit->edit_op[hit->mismatches].pos = 1;
						hit->edit_op[hit->mismatches].mm = 1; 
						hit->mismatches++;
						// update length:
						length++;
				}
			}
			else {	// do not add hit to hit list
				return 0;
			}
			
		}
	}

	// ##########################################################

	// add to new list
	if (*(HIT_LISTS_OPERATOR+length) != NULL) {
		next = *(HIT_LISTS_OPERATOR+length);
		next->last = hit;
		hit->next = next;
		hit->last = NULL;
		*(HIT_LISTS_OPERATOR+length) = hit;
	}
	else {
		hit->last = NULL;
		hit->next = NULL;
		*(HIT_LISTS_OPERATOR+length) = hit;
	}
#ifdef THREADS
	NUM_HITS++;
	HITS_NUM_PER_LENGTH[length]++;
#endif
	if (length > LONGEST_HIT) {
		LONGEST_HIT = length;
	}
	
	return(1);
}


int insert_into_scorelist(HIT* hit, char d)
{
if (DEBUG) { printf("report "); printhit(hit); }
#ifdef THREADS
        while (pthread_mutex_trylock(&hits_into_score_mutex)) usleep(1);
#endif
	
	if (d && duplicate(hit)) {
#ifdef THREADS
                pthread_mutex_unlock(&hits_into_score_mutex);
#endif
		return 0;
	}
	int interval = (hit->mismatches-hit->gaps) * MM_SCORE + hit->gaps * GAP_SCORE - (READ_LENGTH-hit->mismatches) * M_SCORE;

	if (HITS_BY_SCORE[interval].num == 0) {
		// first entry in list
		HITS_BY_SCORE[interval].hitpointer = hit;
	}
	else {
		// list has already some entries, insert to the front
		HIT *tmp_hit = HITS_BY_SCORE[interval].hitpointer;
		HITS_BY_SCORE[interval].hitpointer = hit;
		hit->same_eo_succ = tmp_hit;
	}
	HITS_BY_SCORE[interval].num++;
	HITS_IN_SCORE_LIST++;

#ifdef THREADS
        pthread_mutex_unlock(&hits_into_score_mutex);

        while (pthread_mutex_trylock(&num_edit_ops_mutex)) usleep(1);
#endif

	if (ALL_HIT_STRATEGY != 1) {
		if (interval < SCORE_LIMIT) {
			if (!SECOND_BEST_HIT_STRATEGY) {
				SCORE_LIMIT = interval;
				NUM_EDIT_OPS = hit->mismatches;
			}
			else if (interval != SCORE_LIMIT) {
				if (interval < BEST_SCORE) {
					SCORE_LIMIT = BEST_SCORE;
					BEST_SCORE = interval;
				}
				else if (interval != BEST_SCORE) {
					SCORE_LIMIT = interval;
					NUM_EDIT_OPS = hit->mismatches;
				}
			}
		}
	}
//if (DEBUG) printf("edit %d  bestscore %d scorelimit %d\n",NUM_EDIT_OPS, BEST_SCORE, SCORE_LIMIT);

#ifdef THREADS
	pthread_mutex_unlock(&num_edit_ops_mutex);
#endif

	return 1;
}


int duplicate(HIT* hit) {
	int hitlength = hit->end - hit->start + 1;
	unsigned int readstart, readend;
	char strand;

	if (hit->orientation == '+') {
		readstart = hit->start - hit->readpos + hit->start_offset;	// start pos of read in genome	   0-initialized
		readend = hit->end + (READ_LENGTH - hit->readpos - hitlength) + hit->end_offset;		// 0-initialized
		strand = 1;
	}
	else {
		readstart = hit->start - (READ_LENGTH - hit->readpos - hitlength + 2) + hit->start_offset; 	// 0-initialized
		readend = hit->end + hit->readpos - 2 + hit->end_offset;					// 0-initialized
		strand = -1;
	}
	
	CHROMOSOME_ENTRY *chromosome_director, *chromosome_director_new;
	MAPPING_ENTRY *mapping_entry, *existing_entry;
	char flag = 0;
	
	if (*(GENOME+readstart) == NULL) {
				flag = 1;
				chromosome_director = alloc_chromosome_entry(&readstart, &(hit->chromosome), &strand);
				if (!chromosome_director) return(0);	// chrom_container_size too small -> cancel this read
				*(GENOME + readstart) = chromosome_director;
	}
	else {
		
		chromosome_director = *(GENOME + readstart);

		// is chrom_director from the actual read or from a former one?
		if (chromosome_director >= (CHROMOSOME_ENTRY_OPERATOR->entries+CHROMOSOME_ENTRY_OPERATOR->used) 
			|| 	(chromosome_director < (CHROMOSOME_ENTRY_OPERATOR->entries+CHROMOSOME_ENTRY_OPERATOR->used) &&
				(chromosome_director->genome_pos != readstart))) {
					
					// it's from a former read, thus we need a new chrom_director:
					chromosome_director = alloc_chromosome_entry(&readstart, &(hit->chromosome), &strand);

					if (!chromosome_director) return(0);	// chrom_container_size too small -> cancel this read
					*(GENOME + readstart) = chromosome_director;
					
		}
	}
	
	
 	// Search for chromosome_director for the correct chromosome
	while (chromosome_director->next != NULL && (chromosome_director->chromosome != hit->chromosome || (chromosome_director->chromosome == hit->chromosome && chromosome_director->strand != strand))) {
		chromosome_director = chromosome_director->next;
	}

	if (chromosome_director->chromosome != hit->chromosome || chromosome_director->strand != strand) {
		chromosome_director_new = alloc_chromosome_entry(&readstart, &(hit->chromosome), &strand);
        	if (!chromosome_director_new) return(0);

		chromosome_director_new->next = chromosome_director->next;
		chromosome_director->next = chromosome_director_new;
		chromosome_director = chromosome_director_new;
	}

	if (chromosome_director->mapping_entries == NULL) {
		mapping_entry = alloc_mapping_entry();
		chromosome_director->mapping_entries = mapping_entry;
		
		mapping_entry->readpos = readend;
		mapping_entry->hit = hit;
	}
	else {
		
		if (flag == 1) {
			printf("!!!!!!!!!!!Found entry in chr dir at genome pos %d  --  %p\n", readstart, chromosome_director->mapping_entries	);
			exit(1);
		}
		
		unsigned char score2, score1 = (hit->mismatches-hit->gaps) * MM_SCORE + hit->gaps * GAP_SCORE - (READ_LENGTH - hit->mismatches) * M_SCORE;
		
		existing_entry = chromosome_director->mapping_entries;

		while (existing_entry != NULL) {
			score2 = (existing_entry->hit->mismatches-existing_entry->hit->gaps) * MM_SCORE
				+ existing_entry->hit->gaps * GAP_SCORE - (READ_LENGTH - existing_entry->hit->mismatches) * M_SCORE;
			
#ifndef METHYLOME
			if (existing_entry->readpos == readend && score1 == score2) {
				return 1;
			}
#else

			if (existing_entry->readpos == readend && score1 == score2) {
				if (existing_entry->hit->conversion == hit->conversion || existing_entry->hit->conversion == 3) {
					return 1;
				}
				else { 
					// All hits without conversions prevent also hits from the other conversion state (also featuring no conversions). 
					// Conversions can only be in the matching regions.
					if (features_conversion(existing_entry->hit) == 0) {
						existing_entry->hit->conversion = 3;
						return 1;
					}
				}
			}
#endif
			
			existing_entry = existing_entry->succ;
		}
		
		// no hit with same end pos of read and same score could have been found -> create another entry:
		existing_entry = chromosome_director->mapping_entries; 
		mapping_entry = alloc_mapping_entry();
		mapping_entry->succ = existing_entry;
		existing_entry->pred = mapping_entry;
		//insert to the front of the mapping list
		chromosome_director->mapping_entries = mapping_entry;
		mapping_entry->readpos = readend;
		mapping_entry->hit = hit;
		
	}
	
	return 0;
}

#ifdef METHYLOME
int features_conversion(HIT* hit) {

	int i, j;
	unsigned int readstart;
	int hitlength = hit->end - hit->start + 1;
	if (hit->orientation == '+') {
		readstart = hit->start - hit->readpos + hit->start_offset;	// start pos of read in genome	0-initialized
	}
	else {
		readstart = hit->start - (READ_LENGTH - hit->readpos - hitlength + 2) + hit->start_offset; 	// 0-initialized
	}


	if (hit->mismatches == 0) {
		if (hit->orientation == '+') {
                	for (i = 0; i < READ_LENGTH; i++) {
				if (*(CHR_SEQ[hit->chromosome]+readstart+i) != READ[i]) {
					return 1;
				}
                        }
                }
                else {
                	for (i = 0; i < READ_LENGTH; i++) {
                        	if (get_compl_base(*(CHR_SEQ[hit->chromosome]+readstart+i)) != READ[READ_LENGTH - i - 1]) {
					return 1;
                                }
                        }
                }
	}	
	else {

		char gap_offset = 0;
		char gap_in_read = 0;
		char gap_in_chr = 0;
		
		// sort mismatches in ascending order according to their abs positions and 'gap before mm'-strategy if equal
		qsort(hit->edit_op, hit->mismatches, sizeof(EDIT_OPS), compare); 
		
		for (j=0; j!=hit->mismatches; j++) {
			
			if (hit->edit_op[j].pos < 0) {
				hit->edit_op[j].pos = -hit->edit_op[j].pos;
				gap_in_chr = 1;
			}
		
			if (j == 0) {
			    if (hit->orientation == '+') {
				    for (i = 0; i < hit->edit_op[0].pos-1; i++) {
					    if (*(CHR_SEQ[hit->chromosome]+readstart+i) != READ[i]) {
						  return 1;
					    }
				    }
			    }
			    else {
				    for (i = 0; i < hit->edit_op[0].pos-1; i++) {
					    if (get_compl_base(*(CHR_SEQ[hit->chromosome]+readstart+i)) != READ[READ_LENGTH - i - 1]) {
						  return 1;
					    }
				    }
			    }

			}
			else if (hit->edit_op[j].pos - hit->edit_op[j-1].pos != 0) {			    
			    if (hit->orientation == '+') {
				    for (i = 0; i < (hit->edit_op[j].pos - hit->edit_op[j-1].pos - 1 + gap_in_read); i++) {
					int locreadpos = i + hit->edit_op[j-1].pos - gap_in_read;
					if (*(CHR_SEQ[hit->chromosome] + readstart + i + hit->edit_op[j-1].pos + gap_offset - gap_in_read) != READ[locreadpos]) {
					    return 1;
					}
				    }
			    }
			    else {
				    for (i = 0; i < (hit->edit_op[j].pos - hit->edit_op[j-1].pos - 1 + gap_in_read); i++) {
					    int locreadpos = READ_LENGTH - hit->edit_op[j-1].pos - i - 1 + gap_in_read;
					    if (*(CHR_SEQ[hit->chromosome] + readstart + i + hit->edit_op[j-1].pos + gap_offset - gap_in_read) != get_compl_base(READ[locreadpos])) {
						return 1;
					    }
				    }
			    }
			}	// else: edit_op[j-1] must have been a gap!
				
			gap_in_read = 0;
				
			if (hit->edit_op[j].mm) {
			
			}
			else if (gap_in_chr) {
				gap_offset--;
				gap_in_chr = 0;
			}
			else {
				gap_offset++;
				gap_in_read = 1;
			}
		}
	
			
		// from last mismatch to end of read:
		if (hit->orientation == '+') {
			for (i = 0; i < READ_LENGTH - hit->edit_op[j-1].pos + gap_in_read; i++) {
				if (*(CHR_SEQ[hit->chromosome] + readstart + i + hit->edit_op[j-1].pos + gap_offset - gap_in_read) != READ[i + hit->edit_op[j-1].pos - gap_in_read]) {
				    return 1;
				}
			}
		}
		else {
			for (i = 0; i < READ_LENGTH - hit->edit_op[j-1].pos + gap_in_read; i++) {
				int locreadpos = READ_LENGTH - hit->edit_op[j-1].pos - i - 1 + gap_in_read;
				int locchrpos = readstart + i + hit->edit_op[j-1].pos + gap_offset - gap_in_read;
				if (get_compl_base(*(CHR_SEQ[hit->chromosome] + locchrpos)) != READ[locreadpos]) {
				    return 1;
				}
			}
		}
	}
	

	return 0;
}
#endif

// for debugging
char *get_seq(unsigned int n)
{
	char *seq = (char *) malloc ((INDEX_DEPTH+1)*sizeof(char));
	int i, c;
	for (i=INDEX_DEPTH-1; i>=0; --i) {
		c = (int) (n / POWER[i]);
		switch (c)
		{
			case 0: seq[i] = 'A';
					break;
			case 1: seq[i] = 'C';
					break;
			case 2: seq[i] = 'G';
					break;
			case 3: seq[i] = 'T';
					break;
		}
		n -= (int) (c * POWER[i]);
	}
	seq[INDEX_DEPTH] = '\0';
	return seq;
}

//SG
char unique_base(char c)
{
        return (c == 'A' || c == 'C' || c == 'G' || c == 'T');
}
