// Authors: Korbinian Schneeberger and Joerg Hagmann
// Copyright (C) 2008 by Max-Planck Institute for Developmental Biology, Tuebingen, Germany

#include "genomemapper.h"


#ifndef THREADS
int kbound_overhang_alignment(HIT* hit, int offset, int readseq_start);
int kbound_global_alignment(HIT* hit);


int align_hit_simple(HIT* hit)
{
	int hitlength = hit->end - hit->start + 1;
	int afterhit_len = READ_LENGTH - hit->readpos - hitlength + 1;
	
	// checking if read fits on the start or end of chromosome:
	//@TODO small overlaps should be mapped
	if (hit->orientation == '+') {
		if (hit->start < hit->readpos) {
			return 0;
		}
		if (hit->end + afterhit_len > CHR_LENGTH[hit->chromosome]) {
			return 0;
		}
	}
	
	if (hit->orientation == '-') {
		if (afterhit_len >= hit->start) {
			return 0;
		}
		if (hit->end + hit->readpos - 1 > CHR_LENGTH[hit->chromosome]) {
			return 0;
		}
	}
	
	int j;
	
	int readstart;
	if (hit->orientation == '+') {
		readstart = hit->start - hit->readpos;	
	}
	else {
		readstart = hit->start - afterhit_len - 1;	
	}
	
	// from read[0] to read[hit->readpos]
	for (j=0; j!=hit->readpos-1; ++j) {
		
#ifndef METHYLOME
                if ( (hit->orientation == '+') && ((CHR_SEQ[hit->chromosome][hit->start - hit->readpos + j] != READ[j]) || !(unique_base(READ[j]))))
#else

                if ( hit->orientation == '+' && (
                        ((hit->conversion == 1 && CHR_SEQ[hit->chromosome][hit->start - hit->readpos + j] != READ[j] && !(CHR_SEQ[hit->chromosome][hit->start - hit->readpos + j] == 'C' && READ[j] == 'T'))
                        ||
                        (hit->conversion == 2 && CHR_SEQ[hit->chromosome][hit->start - hit->readpos + j] != READ[j] && !(CHR_SEQ[hit->chromosome][hit->start - hit->readpos + j] == 'G' && READ[j] == 'A'))
                        ||
                        !(unique_base(READ[j])))))
#endif
		{
			// create mismatch:
			if (hit->mismatches < NUM_EDIT_OPS) {
				(hit->edit_op[hit->mismatches]).pos = j+1;
				(hit->edit_op[hit->mismatches]).mm = 1;
			}
	
			hit->mismatches++;
		}
			
#ifndef METHYLOME
                if ((hit->orientation == '-') && ((get_compl_base(CHR_SEQ[hit->chromosome][hit->end + hit->readpos - 2 - j]) != READ[j]) || !(unique_base(READ[j]))))
#else
                if (hit->orientation == '-' &&
                        ((hit->conversion == 1 && get_compl_base(CHR_SEQ[hit->chromosome][hit->end + hit->readpos - 2 - j]) != READ[j] && !(get_compl_base(CHR_SEQ[hit->chromosome][hit->end + hit->readpos - 2 - j]) == 'C' && READ[j] == 'T'))
                        ||
                        (hit->conversion == 2 && get_compl_base(CHR_SEQ[hit->chromosome][hit->end + hit->readpos - 2 - j]) != READ[j] && !(get_compl_base(CHR_SEQ[hit->chromosome][hit->end + hit->readpos - 2 - j]) == 'G' && READ[j] == 'A'))
                        ||
                        !(unique_base(READ[j]))))
#endif
		{
			// create mismatch:
			if (hit->mismatches < NUM_EDIT_OPS) {
				hit->edit_op[hit->mismatches].pos = READ_LENGTH - j;
				hit->edit_op[hit->mismatches].mm = 1;
			}
	
			hit->mismatches++;
		}
		
		if (hit->mismatches > NUM_EDIT_OPS) {
			return 0;
		}
	}
	
	// from read[hit->readpos + hitlength] to read[READ_LENGTH - 1]
	int i = 0;
	j = hit->readpos + hitlength - 1;
	
	while ((hit->mismatches <= NUM_EDIT_OPS) && (j < READ_LENGTH)) {

#ifndef METHYLOME
                if ((hit->orientation == '+') && ((CHR_SEQ[hit->chromosome][hit->end + i] != READ[j]) || !(unique_base(READ[j]))))
#else
                if ((hit->orientation == '+') &&
                        ((hit->conversion == 1 && CHR_SEQ[hit->chromosome][hit->end + i] != READ[j] && !(CHR_SEQ[hit->chromosome][hit->end + i] == 'C' && READ[j] == 'T'))
                        ||
                        (hit->conversion == 2 && CHR_SEQ[hit->chromosome][hit->end + i] != READ[j] && !(CHR_SEQ[hit->chromosome][hit->end + i] == 'G' && READ[j] == 'A'))
                        ||
                        !(unique_base(READ[j]))))
#endif
		{
			// create mismatch:
			if (hit->mismatches < NUM_EDIT_OPS) {		
				(hit->edit_op[hit->mismatches]).pos = j+1;
				(hit->edit_op[hit->mismatches]).mm = 1;
			}
	
			hit->mismatches++;
		}

			
#ifndef METHYLOME
                if (    (hit->orientation == '-') && ((get_compl_base(CHR_SEQ[hit->chromosome][hit->start - 2 - i]) != READ[j]) || !(unique_base(READ[j]))))
#else
                if ((hit->orientation == '-') &&
                        ((hit->conversion == 1 && get_compl_base(CHR_SEQ[hit->chromosome][hit->start - 2 - i]) != READ[j] && !(get_compl_base(CHR_SEQ[hit->chromosome][hit->start - 2 - i]) == 'C' &&  READ[j] == 'T'))
                        ||
                        (hit->conversion == 2 && get_compl_base(CHR_SEQ[hit->chromosome][hit->start - 2 - i]) != READ[j] && !(get_compl_base(CHR_SEQ[hit->chromosome][hit->start - 2 - i]) == 'G' && READ[j] == 'A'))
                        ||
                        !(unique_base(READ[j]))))
#endif
		{
			// create mismatch:
			if (hit->mismatches < NUM_EDIT_OPS) {
				(hit->edit_op[hit->mismatches]).pos = READ_LENGTH - j;
				(hit->edit_op[hit->mismatches]).mm = 1;
			}
	
			hit->mismatches++;
		}
		
		if (hit->mismatches > NUM_EDIT_OPS) {
			return 0;
		}
		
		++i;
		++j;
	}
	
	if (hit->mismatches <= NUM_MISMATCHES) {	// there can't be gaps
		insert_into_scorelist(hit, 1);
		//if (!ALL_HIT_STRATEGY && hit->mismatches < NUM_EDIT_OPS) NUM_EDIT_OPS = hit->mismatches;
	}
	
	return 0;
}


// returns if aligned hit fulfills MM and gap-criterias, thus is printed out (alignments are called in this method)
int prepare_kbound_alignment(HIT* hit)
{
	
	int hitlength = hit->end - hit->start + 1;
	int afterhit_len = READ_LENGTH - hit->readpos - hitlength + 1;
	
	// checking if read fits on the start or end of chromosome:
	//@TODO small overlaps should be mapped
	if (hit->orientation == '+') {
		if (hit->start < hit->readpos) {
			return 0;
		}
		if (hit->end + afterhit_len > CHR_LENGTH[hit->chromosome]) {
			return 0;
		}
	}
	
	if (hit->orientation == '-') {
		if (afterhit_len >= hit->start) {
			return 0;
		}
		if (hit->end + hit->readpos - 1 > CHR_LENGTH[hit->chromosome]) {
			return 0;
		}
	}
	
	
	// just perform global alignment if gap heuristic/speedup was disabled:
	if (!OVERHANG_ALIGNMENT) {
		if (kbound_global_alignment(hit)) {
			//if (!ALL_HIT_STRATEGY && hit->mismatches < NUM_EDIT_OPS) NUM_EDIT_OPS = hit->mismatches;
			return insert_into_scorelist(hit, 1);
		}
		else return 0;
	}
	
	
	// perform whole alignment pipeline:
	
	char k1_aligned;
	int offset;
	
	int readstart;	// start pos of read on genome
	if (hit->orientation == '+') {
		readstart = hit->start - hit->readpos;	
	}
	else {
		readstart = hit->start - afterhit_len - 1;	
	}
	int readend = readstart + READ_LENGTH;	// end pos of read on genome
	
	if (hit->readpos != 1) {
			if (hit->orientation == '+') {
				if (readstart - NUM_GAPS < 0) offset = readstart;
					else offset = NUM_GAPS;
			} 
			else {
				if (readend + NUM_GAPS > CHR_LENGTH[hit->chromosome]) offset = CHR_LENGTH[hit->chromosome] - readend;
					else offset = NUM_GAPS;
			}

			// perform alignment
			k1_aligned = kbound_overhang_alignment(hit, offset, 0);
		
			// there are gaps on best path in alignment -> perform whole global alignment
			if (k1_aligned == 0) {
				
				if (kbound_global_alignment(hit)) {
					//if (!ALL_HIT_STRATEGY && hit->mismatches < NUM_EDIT_OPS) NUM_EDIT_OPS = hit->mismatches;
					return insert_into_scorelist(hit, 1);
				}
				else return 0;
			}
			
			// too many mismatches in aligned read already:	
			if (k1_aligned == -1) {
				return 0;
			}
			
	}
	
	if (hit->readpos + hitlength != READ_LENGTH + 1) {
		
			if (hit->orientation == '+') {
				if (readend + NUM_GAPS > CHR_LENGTH[hit->chromosome]) offset = CHR_LENGTH[hit->chromosome] - readend;
					else offset = NUM_GAPS;
			}
			else {
				if (readstart - NUM_GAPS < 0) offset = readstart;
					else offset = NUM_GAPS;
			}

			// perform alignment if at least one edit op can still be afforded:
			if (hit->mismatches < NUM_EDIT_OPS) {
				k1_aligned = kbound_overhang_alignment(hit, offset, hit->readpos+hitlength-1);
			}
			else {
				return 0;
			}
			
			// there are gaps on best path in alignment -> perform whole global alignment
			if (k1_aligned == 0) {
			
				if (kbound_global_alignment(hit)) {
					//if (!ALL_HIT_STRATEGY && hit->mismatches < NUM_EDIT_OPS) NUM_EDIT_OPS = hit->mismatches;
					return insert_into_scorelist(hit, 1);
				}
				else return 0;
			}
				
			// too many mismatches?
			if (k1_aligned == -1) {
				return 0;
			}
	}
	
	// gapless alignment was successful -> insert hit into HITS_BY_SCORE:
	insert_into_scorelist(hit, 1);
	
	//if (!ALL_HIT_STRATEGY && hit->mismatches < NUM_EDIT_OPS) NUM_EDIT_OPS = hit->mismatches;
	
	
	// successful alignment:
	return 1;
}


// returns 1 if alignment is gapless, 0 if there are gaps and -1 if nr of allowed MMs is exceeded 
int kbound_overhang_alignment(HIT* hit, int offset, int readstart)
{
	int K = NUM_GAPS;

	int length;
	int chrstart;
	char offset_comp = 0;
	
	if (readstart == 0) {
		length = hit->readpos + offset - 1;
		offset_comp = offset;
		if (hit->orientation == '+') chrstart = hit->start - length - 1;
			else chrstart = hit->end + length - 1;
	}
	else {
		length = READ_LENGTH - (hit->end - hit->start) - hit->readpos + offset;
		if (hit->orientation == '+') chrstart = hit->end;
			else chrstart = hit->start - 2;
	}

	int i,j,h;
	
	// Initialization:
	unsigned char score_before = hit->mismatches * MM_SCORE;
	if (readstart == 0) {
		for (i=0; i!=2*K+1; ++i) {
			M[i][offset] = score_before;
			//T[i][offset] = '0';
		}
		j = (K-offset < 0)? 0: K-offset;
		for (i=0; i!=j; ++i) {
			M[0][i+1+offset] = score_before + (i+1) * GAP_SCORE;
			//T[0][i+1+offset] = UP;
		}
	}
	else {
		for (i = 0; i <= K; ++i) {
			M[i][0] = score_before + i * GAP_SCORE;
			if (i!=0) { 
				M[0][i] = score_before + i * GAP_SCORE;
				
				//T[i][0] = LEFT;
				//T[0][i] = UP;
			}
		}
		//T[0][0] = '0';
	}
	
	// Alignment:
	int c, min_i = -1;
	char best_score_on_diag = 0;
	unsigned char score;
	unsigned char best_score = WORST_SCORE + 1;
	unsigned char column_score = best_score;
	
	for (i = 1; i <= length; ++i) {
		
		for (h = -K; h <= K; ++h) {
			j = i + h;
		
				if (j <= length && j >= 1) {
					if (j>K) c = j - K;
						else c = 0;
					
				if ( !(readstart == 0 && j <= offset) && !(readstart != 0 && j > length-offset) ) {
					
					// Score-Function:
					if (hit->orientation == '+') {
#ifndef METHYLOME
                                                if (CHR_SEQ[hit->chromosome][chrstart+i-1] != READ[readstart+j-offset_comp-1] || !unique_base(READ[readstart+j-offset_comp-1]))
#else
                                                if (    (hit->conversion == 1 && CHR_SEQ[hit->chromosome][chrstart+i-1] != READ[readstart+j-offset_comp-1] && !(CHR_SEQ[hit->chromosome][chrstart+i-1] == 'C' && READ[readstart+j-offset_comp-1] == 'T'))
                                                        ||
                                                        (hit->conversion == 2 && CHR_SEQ[hit->chromosome][chrstart+i-1] != READ[readstart+j-offset_comp-1] && !(CHR_SEQ[hit->chromosome][chrstart+i-1] == 'G' && READ[readstart+j-offset_comp-1] == 'A'))
                                                        ||
                                                        !unique_base(READ[readstart+j-offset_comp-1]))
#endif
                                                {
                                                         score = MM_SCORE;
                                                }
                                                else {
                                                        score = M_SCORE;
                                                }
					}
					else {
#ifndef METHYLOME
                                                if (get_compl_base(CHR_SEQ[hit->chromosome][chrstart-i+1]) != READ[readstart+j-offset_comp-1] || !unique_base(READ[readstart+j-offset_comp-1]))
#else
                                                if (    (hit->conversion == 1 && get_compl_base(CHR_SEQ[hit->chromosome][chrstart-i+1]) != READ[readstart+j-offset_comp-1] && !(get_compl_base(CHR_SEQ[hit->chromosome][chrstart-i+1]) == 'C' && READ[readstart+j-offset_comp-1] == 'T'))
                                                        ||
                                                        (hit->conversion == 2 && get_compl_base(CHR_SEQ[hit->chromosome][chrstart-i+1]) != READ[readstart+j-offset_comp-1] && !(get_compl_base(CHR_SEQ[hit->chromosome][chrstart-i+1]) == 'G' && READ[readstart+j-offset_comp-1] == 'A'))
                                                        ||
                                                        !unique_base(READ[readstart+j-offset_comp-1]))
#endif
                                                {
                                                         score = MM_SCORE;
                                                }
                                                else {
                                                        score = M_SCORE;
                                                }
					}
					
					M[i-c][j] = M[i-c-(j<=K)][j-1] + score;
					
					if ((i-j-1 <= K) && (i-j-1 >= -K)) {
						if (M[i-c-1][j] + GAP_SCORE <= M[i-c][j]) {
							
							// gap encountered
							if (i == j) {
								return 0;
							}
							
							M[i-c][j] = M[i-c-1][j] + GAP_SCORE;
							
						}
					}
					if ((i-j+1 <= K) && (i-j+1 >= -K)) {
						if (M[i-c+(j>K)][j-1] + GAP_SCORE <= M[i-c][j]) {
							
							if (i == j) {
								return 0;
							}
							
							M[i-c][j] = M[i-c+(j>K)][j-1] + GAP_SCORE;
							//T[i-c][j] = UP;	// traceback to upper
							
						}
					}
					
					if (readstart != 0 && j == length - offset) {
						if ( (!best_score_on_diag && M[i-c][j] < best_score) || (best_score_on_diag && M[i-c][j] <= best_score) ) {	
							// ensures that if score of most right bottom cell is equal to another cell in the same row, this other cell is reported with the best score
							best_score = M[i-c][j];
							min_i = i;
							if (i == j) best_score_on_diag = 1;
						}
					}
					
					if (M[i-c][j] < column_score) column_score = M[i-c][j];
					
				}
			}
			
		} //for h
		
		if (column_score > WORST_SCORE) {
			
			// readstart==0: score in most right bottom cell cannot become less than worst score
			// best_score > WORST_SCORE: there is no cell in the bottom row which has a valid alignment score 
			if (readstart == 0 || best_score > WORST_SCORE) {
				return -1;
			} 
			// if best_score has been found before most right bottom cell was processed
			/*else if (i < length - offset) {	// readstart here is != 0
				GAPS_ENCOUNTERED[1]++;
				return 0;
			}*/
			
			break;
			
		} else {
			column_score = WORST_SCORE + 1;
		}
		
	} //for i
	
	
	if (readstart == 0) {
		j = length;
		i = K - (j<K) * (K - j);
		best_score = M[i][j];
		min_i = j;
	}
	else {
		j = length-offset;
		i = min_i - (j>K) * (j - K);
	}
	
	if (best_score > WORST_SCORE) {
		return -1;
	}
	
	// if best score is worse than max nr of allowed mismatches...
	if (best_score > WORST_MM_SCORE) {
		// ...and best score is in the most right bottom cell, there is no better path with gaps -> so we have too many MMs
		if (min_i == j) {
			return -1;
		}
		// ..and there is another cell with this score other than the most right bottom cell -> there is a path with a valid score, BUT with gaps! 
		else {
			return 0;
		}
	}
	
	// FOR READSTART != 0 ONLY: if best path doesn't start from the most right bottom corner of the matrix -> gaps are in best alignment, return 0
	//if (min_i != j && M[i][j] != M[K-(j<K)*(K-j)][j]) {	@TODO!!!!!!!
	if (min_i != j) {	// even if score in most right bottom cell is equal to best score -> path with gaps needs less edit ops -> perform global algnm.
		return 0;
	}
	// FOR READSTART == 0: traceback has to start in most right bottom corner

	

	// Traceback (if there had been gaps the procedure returned 0, so only mismatches are left -> only diagonal traceback)
	int readpos;
	if (readstart == 0) readpos = j - offset;
		else readpos = readstart + j;
	i = K - (j<K) * (K - j);

	int mms = (int) (M[i][j] / MM_SCORE) - (int) (score_before / MM_SCORE);

	while (j != (readstart==0)*offset && mms != 0) {
		if (M[i][j] != M[i-(j<=K)][j-1]) {
			// create mismatch:
			
			if (hit->orientation == '+') hit->edit_op[hit->mismatches].pos = readpos;
				else		     hit->edit_op[hit->mismatches].pos = READ_LENGTH - readpos + 1;
			hit->edit_op[hit->mismatches].mm = 1;
			hit->mismatches++;
			
			--mms;
		}
		--readpos;
		--j;
		if (j<K) --i;
	}

	// successfully aligned
	return 1;
}






// k-bound global alignment algorithm:
int kbound_global_alignment(HIT* hit)
{
	
	int K = NUM_GAPS;
	
	// delete possible mismatches found in k1_alignment or at the beg/end:
	hit->mismatches = 0;

	int chrstart, chrend, offset_front, offset_end;
	if (hit->orientation == '+') {
		chrstart = hit->start - hit->readpos;		
		if (chrstart < K) offset_front = chrstart;
			else offset_front = K;
		
		chrend = chrstart + READ_LENGTH;	
		if (chrend + K > CHR_LENGTH[hit->chromosome]) offset_end = CHR_LENGTH[hit->chromosome] - chrend;
			else offset_end = K;
		
		chrstart -= offset_front;
	}
	else {
		chrstart = hit->end + hit->readpos - 2;		
		if (chrstart + K >= CHR_LENGTH[hit->chromosome]) offset_front = CHR_LENGTH[hit->chromosome] - chrstart - 1;
			else offset_front = K;
		
		chrend = chrstart - READ_LENGTH + 1;	
		if (chrend - K < 0) offset_end = chrend;
			else offset_end = K;
			
		chrstart += offset_front;	
	}

	int length = READ_LENGTH + offset_front + offset_end;
	



	int i,j,h;
	
	// Initialization:
	for (i=0; i!=2*K+1-(K-offset_front); ++i) {
		
		M[i][offset_front] = 0;
		T[i][offset_front] = '0';
		
		j = (K-offset_front < 0)? 0: K-offset_front;
		for (h=0; h!=j; ++h) {
			M[0][h+1+offset_front] = (h+1) * GAP_SCORE;
			T[0][h+1+offset_front] = UP;
		}
	}
	
	
	// Alignment:
	int c;
	unsigned char best_score = WORST_SCORE + 1;
	unsigned char column_score = best_score;
	unsigned score;
	int min_i = 0;	// start of traceback
	char best_score_on_diag = 0;
	
	for (i = 1; i <= length; ++i) {
		
		for (h = -K; h <= K; ++h) {
			j = i + h;
		
				if (j <= length && j >= 1) {
					if (j>K) c = j - K;
						else c = 0;
					
				if ( j > offset_front && j <= length-offset_end ) {
					
					// Score-Function:
					if (hit->orientation == '+') {
#ifndef METHYLOME
                                                if (CHR_SEQ[hit->chromosome][chrstart+i-1] != READ[j-offset_front-1] || !unique_base(READ[j-offset_front-1]))
#else
                                                if (    (hit->conversion == 1 && CHR_SEQ[hit->chromosome][chrstart+i-1] != READ[j-offset_front-1] && !(CHR_SEQ[hit->chromosome][chrstart+i-1] == 'C' && READ[j-offset_front-1] == 'T'))
                                                        ||
                                                        (hit->conversion == 2 && CHR_SEQ[hit->chromosome][chrstart+i-1] != READ[j-offset_front-1] && !( CHR_SEQ[hit->chromosome][chrstart+i-1] == 'G' && READ[j-offset_front-1] == 'A'))
                                                        ||
                                                        !unique_base(READ[j-offset_front-1]))
#endif
                                                {
                                                         score = MM_SCORE;
                                                }
                                                else {
                                                        score = M_SCORE;
                                                }
					}
					else {
#ifndef METHYLOME
                                                if (get_compl_base(CHR_SEQ[hit->chromosome][chrstart-i+1]) != READ[j-offset_front-1] || !unique_base(READ[j-offset_front-1])) //COMP
#else
                                                if (    (hit->conversion == 1 && get_compl_base(CHR_SEQ[hit->chromosome][chrstart-i+1]) != READ[j-offset_front-1] && !(get_compl_base(CHR_SEQ[hit->chromosome][chrstart-i+1]) == 'C' && READ[j-offset_front-1] == 'T'))
                                                        ||
                                                        (hit->conversion == 2 && get_compl_base(CHR_SEQ[hit->chromosome][chrstart-i+1]) != READ[j-offset_front-1] && !(get_compl_base(CHR_SEQ[hit->chromosome][chrstart-i+1]) == 'G' && READ[j-offset_front-1] == 'A'))
                                                        ||
                                                        !unique_base(READ[j-offset_front-1])) //COMP
#endif
                                                {
                                                         score = MM_SCORE;
                                                }
                                                else {
                                                         score = M_SCORE;
                                                }
					}
					
					M[i-c][j] = M[i-c-(j<=K)][j-1] + score;
					T[i-c][j] = DIAGONAL;	// traceback diagonally
					
					if ((i-j+1 <= K) && (i-j+1 >= -K)) {
						// gap in chr
						if (M[i-c+(j>K)][j-1] + GAP_SCORE == M[i-c][j] && 
								((hit->orientation == '+' && GAPS_MOST_RIGHT) || (hit->orientation == '-' && !GAPS_MOST_RIGHT))) {
							M[i-c][j] = M[i-c+(j>K)][j-1] + GAP_SCORE;
							T[i-c][j] = UP;	// traceback to upper with gaps most right
						}
						else if (M[i-c+(j>K)][j-1] + GAP_SCORE < M[i-c][j]) {
							M[i-c][j] = M[i-c+(j>K)][j-1] + GAP_SCORE;
							T[i-c][j] = UP;	// traceback to upper
						}
					}
					
					if ((i-j-1 <= K) && (i-j-1 >= -K)) {
						// gap in read
						if (M[i-c-1][j] + GAP_SCORE == M[i-c][j] && 
								((hit->orientation == '+' && GAPS_MOST_RIGHT) || (hit->orientation == '-' && !GAPS_MOST_RIGHT))) {
							M[i-c][j] = M[i-c-1][j] + GAP_SCORE;
							T[i-c][j] = LEFT;	// traceback to the left with gaps most right
						}
						else if (M[i-c-1][j] + GAP_SCORE < M[i-c][j]) {
							M[i-c][j] = M[i-c-1][j] + GAP_SCORE;
							T[i-c][j] = LEFT;	// traceback to the left
						}
					}

					// Remember best score, i.e. start of traceback
					if (j == length - offset_end) {
						
						// gaps in reads preferred to gaps in chr:
						if ( (i <= j && M[i-c][j] <= best_score) || (i > j && ((best_score_on_diag && M[i-c][j] <  best_score) 
											 || (!best_score_on_diag && M[i-c][j] <= best_score))) ) {
						
							best_score = M[i-c][j];
							min_i = i;
							if (i == j) best_score_on_diag = 1;
						}
					}
					// best_score preference:
					//	1. diagonal
					//	2. i > j (gaps in reads)
					//	3. i < j (gaps in chr)
					
					if (M[i-c][j] < column_score) column_score = M[i-c][j];
						
				}
			}
			
		} //for h
		
		if (column_score > WORST_SCORE) {
			if (best_score > WORST_SCORE) {
				return 0;
			}
			else {
				break;
			}
		} else {
			column_score = WORST_SCORE + 1;
		}
		
	} //for i
	
	
	if (best_score > WORST_SCORE) {
        	return 0;
	}


	j = length - offset_end;
	i = min_i - (j>K)*(j-K);
	int chrpos;
	if (hit->orientation == '+') {
		chrpos = chrstart + min_i - 1;
		hit->end_offset += min_i - j;	// correcting the read pos on chr for later output
	}
	else {
		chrpos = chrstart - min_i + 1;
		hit->start_offset -= min_i - j;	// correcting the read pos on chr for later output
	}
	int readpos = READ_LENGTH - 1;
	
	
	
	
	// Traceback:
	while (T[i][j] != '0' && M[i][j] != 0) {
			
		switch (T[i][j]) {		
			case DIAGONAL: {
				
				//if (CHR_SEQ[hit->chromosome][chrpos] != READ[readpos] || !unique_base(READ[readpos])) {
				//if (M[i-(j<=K)][j-1] == M[i][j] - MM_SCORE) {					// doesn't work for doubles!!
				//if (fabs(M[i-(j<=K)][j-1] - M[i][j] + MM_SCORE) > 1E-9) {		// works, but requires math.h!
				if (M[i-(j<=K)][j-1] != M[i][j]) {
					if (hit->mismatches < NUM_EDIT_OPS && (hit->mismatches-hit->gaps) < NUM_MISMATCHES) {
						if (hit->orientation == '+') hit->edit_op[hit->mismatches].pos = readpos + 1;
							else		     hit->edit_op[hit->mismatches].pos = READ_LENGTH - readpos;
						hit->edit_op[hit->mismatches].mm = 1;
						hit->mismatches++; 
					}
					else {
						return 0;	// discard hit
					}
				}
				
				i = i-(j<=K);
				j--;
				if (hit->orientation == '+') chrpos--;
					else chrpos++;
				readpos--;
				break;
			}
			
			case LEFT: {
		
				if (hit->mismatches < NUM_EDIT_OPS && (!STRINGENT_GAPLIMIT || hit->gaps < NUM_GAPS)) {
					if (hit->orientation == '+') hit->edit_op[hit->mismatches].pos = readpos + 2;
						else		     hit->edit_op[hit->mismatches].pos = READ_LENGTH - readpos;
					hit->edit_op[hit->mismatches].mm = 0;
					hit->mismatches++;
					hit->gaps++;
				}
				else {
					return 0;	// discard hit
				}
					
				i--;
				if (hit->orientation == '+') chrpos--;
					else chrpos++;
				break;
			}
			
			case UP: {
				
				if (hit->mismatches < NUM_EDIT_OPS && (!STRINGENT_GAPLIMIT || hit->gaps < NUM_GAPS)) {
					if (hit->orientation == '+') hit->edit_op[hit->mismatches].pos = -readpos - 1;
						else					 hit->edit_op[hit->mismatches].pos = -READ_LENGTH + readpos; 
					hit->edit_op[hit->mismatches].mm = 0;
					hit->mismatches++;
					hit->gaps++;
				}
				else {
					return 0;	// discard hit
				}
				
				i = i+(j>K);
				j--;
				readpos--;
				break;
			}
		}
		
	}
	
	if (hit->orientation == '+') hit->start_offset += i + (j>K)*(j-K) - j;
		else hit->end_offset -= i + (j>K)*(j-K) - j;
	
	// successfully aligned
	return 1;
}

#endif
