// Authors: Korbinian Schneeberger and Joerg Hagmann
// Copyright (C) 2008 by Max-Planck Institute for Developmental Biology, Tuebingen, Germany

#include "genomemapper.h"
#include <sys/time.h>

#ifdef THREADS
int kbound_overhang_alignment(HIT* hit, THREAD_CONST* thread_const, int offset, int readseq_start, int num_thread);
int kbound_global_alignment(HIT* hit, THREAD_CONST* thread_const, int num_thread);


void *prepare_align(void *arg)
{
	long num_thread;
	num_thread = (long) arg;
	int started = 0;
	
	while (NEW_READ) {

	    while (pthread_mutex_trylock(&start_align_mutex[num_thread])) usleep(1);
	    if (!started) {
		pthread_cond_wait(&start_align_cond[num_thread], &start_align_mutex[num_thread]);
		if (!NEW_READ)
			break;
		THREAD_STARTED[num_thread] = 1;
		HIT* hit = (HIT*) malloc (sizeof(HIT));
		int i, j;
		int hitlength;

		for (i=thread_data[num_thread]->hitlength_start; i >= thread_data[num_thread]->hitlength_end; i--)
		{	
			if ((NUM_EDIT_OPS == 0) && (i < thread_data[num_thread]->thread_const->read_length)) 
				break;

       			if ((*(HIT_LISTS_OPERATOR + i)) != NULL) {

                		hit = *(HIT_LISTS_OPERATOR + i);
				j = 1;
				while ((hit !=NULL) && (j < thread_data[num_thread]->hitdepth_start[i])) {
					j++;
					hit = hit->next;
				}
			        // foreach hit with hitlength i:
                       		while ((hit != NULL) && (j <= thread_data[num_thread]->hitdepth_end[i]))  
				{
                               		hitlength = hit->end - hit->start + 1;
                               		// Hit spans the whole read:
                               		if (hitlength == thread_data[num_thread]->thread_const->read_length) {

                                       		// report match:
                                       		if (hit->mismatches <= NUM_EDIT_OPS) {

                                               		// insert hit into HITS_BY_SCORE
                                               		hit->aligned = 1;
							insert_into_scorelist(hit, 1);
                                               		if (!thread_data[num_thread]->thread_const->all_hit_strategy && hit->mismatches < NUM_EDIT_OPS)
							{	 
								while (pthread_mutex_trylock(&num_edit_ops_mutex)) 
									usleep(1);
								NUM_EDIT_OPS = hit->mismatches;
								pthread_mutex_unlock(&num_edit_ops_mutex);
							}
                                       		}
                               		}
                       			else {


                                	// Alignment

                                		if (hit->mismatches < NUM_EDIT_OPS) {
                                       			if (thread_data[num_thread]->thread_const->num_gaps != 0) {
                                               		// KBOUND:
								
                                               			prepare_kbound_alignment(hit, thread_data[num_thread]->thread_const, (int)num_thread);
								if (hit->aligned)
									insert_into_scorelist(hit, 1);
				
                                       			}
                                       			else {
                                               		// SIMPLE:
                                               			align_hit_simple(hit, thread_data[num_thread]->thread_const, (int)num_thread);
								if (hit->aligned)
									insert_into_scorelist(hit, 1);
                                       			}
                                		}

                               		} // else has mismatches

                               		hit = hit->next;
					j++;
               			} // while hits in [hitdepth_start, hitdepth_end] 

       			}
		} //for hitlength
	    } //if started
		
	    pthread_mutex_unlock(&start_align_mutex[num_thread]);
		
	    while (pthread_mutex_trylock(&aligned_mutex)) usleep(1);
	    THREADS_FINISHED++;
	    if (THREADS_FINISHED == NUM_THREADS)
		    pthread_cond_signal(&aligned_cond);
	    pthread_mutex_unlock(&aligned_mutex);
	} //while NEW_READ 
	pthread_mutex_unlock(&start_align_mutex[num_thread]);
	pthread_exit(NULL);
}

int align_hit_simple(HIT* hit, THREAD_CONST* thread_const, int num_thread)
{
	int hitlength = hit->end - hit->start + 1;
	int afterhit_len = thread_const->read_length - hit->readpos - hitlength + 1;
	
	// checking if read fits on the start or end of chromosome:
	//@TODO small overlaps should be mapped
	if (hit->orientation == '+') {
		if (hit->start < hit->readpos) {
			//printf("Read cannot be mapped on plus strand at start of chromosome %d!\n", hit->chromosome+1);
			return 0;
		}
		if (hit->end + afterhit_len > CHR_LENGTH[hit->chromosome]) {
			//printf("Read cannot be mapped on plus strand at end of chromosome %d!\n", hit->chromosome+1);
			return 0;
		}
	}
	
	if (hit->orientation == '-') {
		if (afterhit_len >= hit->start) {
			//printf("Read cannot be mapped on minus strand at start of chromosome %d!\n", hit->chromosome+1);
			return 0;
		}
		if (hit->end + hit->readpos - 1 > CHR_LENGTH[hit->chromosome]) {
			//printf("Read cannot be mapped on minus strand at end of chromosome %d!\n", hit->chromosome+1);
			return 0;
		}
	}
	
	
	int j;
	//int val;
	
	int readstart;
	if (hit->orientation == '+') {
		readstart = hit->start - hit->readpos;	// 0-initialized
	}
	else {
		readstart = hit->start - afterhit_len - 1;	//changed		0-initialized
	}
	
	// from read[0] to read[hit->readpos]
	for (j=0; j!=hit->readpos-1; ++j) {
		
		if (	(hit->orientation == '+') 
				&& (	(CHR_SEQ[hit->chromosome][hit->start - hit->readpos + j] != READ[j]) 
				 	 || !(unique_base(READ[j]))		// [XX] should also be a mismatch!
					// if read[j]=X and chr_seq not, then first or-condition is automatically true -> genome sequence doesn't have to be checked
				   ) 
		   ) 
		{
			// create mismatch:
			if (hit->mismatches < NUM_EDIT_OPS) {
				(hit->edit_op[hit->mismatches]).pos = j+1;
				(hit->edit_op[hit->mismatches]).mm = 1;
			}
	
			hit->mismatches++;
		}

		
	
		if (	(hit->orientation == '-') 
			 	&& (    (get_compl_base(CHR_SEQ[hit->chromosome][hit->end + hit->readpos - 2 - j]) != READ[j]) 
					 || !(unique_base(READ[j]))
				   )
		   )
		{
			// create mismatch:
			if (hit->mismatches < NUM_EDIT_OPS) {
				hit->edit_op[hit->mismatches].pos = thread_const->read_length - j;
				hit->edit_op[hit->mismatches].mm = 1;
			}
	
			hit->mismatches++;
		}

		if (hit->mismatches > NUM_EDIT_OPS) {
			return 0;
		}
	}
	
	// from read[hit->readpos + hitlength] to read[READ_LENGTH - 1]
	int i = 0;
	j = hit->readpos + hitlength - 1;
	
	while ((hit->mismatches <= NUM_EDIT_OPS) && (j < thread_const->read_length)) {

		if (	(hit->orientation == '+') 
			 	&& (    (CHR_SEQ[hit->chromosome][hit->end + i] != READ[j]) 
					 || !(unique_base(READ[j]))		// [XX] should also be a mismatch!
					// if read[j]=X and chr_seq not, then first or-condition is automatically true -> genome sequence doesn't have to be checked
				   ) 
		   )
		{
			// create mismatch:
			if (hit->mismatches < NUM_EDIT_OPS) {		
				(hit->edit_op[hit->mismatches]).pos = j+1;
				(hit->edit_op[hit->mismatches]).mm = 1;
			}
	
			hit->mismatches++;
		}

			
		if (	(hit->orientation == '-') 
			 	&& (    (get_compl_base(CHR_SEQ[hit->chromosome][hit->start - 2 - i]) != READ[j]) 
					 || !(unique_base(READ[j]))
				   )
		   )
		{
			// create mismatch:
			if (hit->mismatches < NUM_EDIT_OPS) {
				(hit->edit_op[hit->mismatches]).pos = thread_const->read_length  - j;
				(hit->edit_op[hit->mismatches]).mm = 1;
			}
	
			hit->mismatches++;
		}
		
		if (hit->mismatches > NUM_EDIT_OPS) {
			return 0;
		}
		
		++i;
		++j;
	}
	
	if (hit->mismatches <= thread_const->num_mismatches) {	// there can't be gaps
		
		// hit aligned
		hit->aligned = 1;
		while (pthread_mutex_trylock(&num_edit_ops_mutex)) usleep(1);
		if (!thread_const->all_hit_strategy && hit->mismatches < NUM_EDIT_OPS) 
			NUM_EDIT_OPS = hit->mismatches;
		pthread_mutex_unlock(&num_edit_ops_mutex);
	}
	

	return 1;
}


// returns if aligned hit fulfills MM and gap-criterias, thus is printed out (alignments are called in this method)
int prepare_kbound_alignment(HIT* hit, THREAD_CONST* thread_const, int num_thread)
{
	int hitlength = hit->end - hit->start + 1;
	int afterhit_len = thread_const->read_length - hit->readpos - hitlength + 1;

	// checking if read fits on the start or end of chromosome:
	//@TODO small overlaps should be mapped
	if (hit->orientation == '+') {
		if (hit->start < hit->readpos) {
			return 0;
		}
		if (hit->end + afterhit_len > CHR_LENGTH[hit->chromosome]) {
			return 0;
		}
	}
	
	if (hit->orientation == '-') {
		if (afterhit_len >= hit->start) {
			return 0;
		}
		if (hit->end + hit->readpos - 1 > CHR_LENGTH[hit->chromosome]) {
			return 0;
		}
	}
	
	
	// just perform global alignment if gap heuristic/speedup was disabled:
	if (!thread_const->overhang_alignment) {
		if (kbound_global_alignment(hit, thread_const, num_thread)) {
			while (pthread_mutex_trylock(&num_edit_ops_mutex)) usleep(1);
			if (!thread_const->all_hit_strategy && hit->mismatches < NUM_EDIT_OPS) 
				NUM_EDIT_OPS = hit->mismatches;
			pthread_mutex_unlock(&num_edit_ops_mutex);
			hit->aligned = 1;
		}
		return 0;
	}
	
	
	// perform whole alignment pipeline:
	
	char k1_aligned;
	int offset;
	
	int readstart;	// start pos of read on genome
	if (hit->orientation == '+') {
		readstart = hit->start - hit->readpos;		//	0-initialized
	}
	else {
		readstart = hit->start - afterhit_len - 1;	//changed		0-initialized
	}
	int readend = readstart + thread_const->read_length;	// end pos of read on genome	1-initialized
	
	if (hit->readpos != 1) {
		
		
			if (hit->orientation == '+') {
				if (readstart - thread_const->num_gaps < 0) offset = readstart;
				else offset = thread_const->num_gaps;
			} 
			else {
				if (readend + thread_const->num_gaps > CHR_LENGTH[hit->chromosome]) 
					offset = CHR_LENGTH[hit->chromosome] - readend;
					else offset = thread_const->num_gaps;
			}

			// perform alignment
			k1_aligned = kbound_overhang_alignment(hit, thread_const, offset, 0, num_thread);
		
			// there are gaps on best path in alignment -> perform whole global alignment
			if (k1_aligned == 0) {
							
				if (kbound_global_alignment(hit, thread_const, num_thread)) {
					while (pthread_mutex_trylock(&num_edit_ops_mutex)) usleep(1);
					if (!thread_const->all_hit_strategy && hit->mismatches < NUM_EDIT_OPS) 
						NUM_EDIT_OPS = hit->mismatches;
					pthread_mutex_unlock(&num_edit_ops_mutex);
					hit->aligned = 1;
				}
				return 0;
			}
			
			// too many mismatches in aligned read already:	
			if (k1_aligned == -1) {
				return 0;
			}
			
	}
	
	if (hit->readpos + hitlength != thread_const->read_length + 1) {
		
			if (hit->orientation == '+') {
				if (readend + thread_const->num_gaps > CHR_LENGTH[hit->chromosome]) 
					offset = CHR_LENGTH[hit->chromosome] - readend;
					else offset = thread_const->num_gaps;
			}
			else {
				if (readstart - thread_const->num_gaps < 0) offset = readstart;
				else offset = thread_const->num_gaps;
			}

			// perform alignment if at least one edit op can still be afforded:
			if (hit->mismatches < NUM_EDIT_OPS)
				k1_aligned = kbound_overhang_alignment(hit, thread_const, offset, hit->readpos+hitlength-1, num_thread);
			else {
				return 0;
			}
			
			// there are gaps on best path in alignment -> perform whole global alignment
			if (k1_aligned == 0) {
			
				if (kbound_global_alignment(hit, thread_const, num_thread)) {
					while (pthread_mutex_trylock(&num_edit_ops_mutex)) usleep(1);
					if (!thread_const->all_hit_strategy && hit->mismatches < NUM_EDIT_OPS) 
						NUM_EDIT_OPS = hit->mismatches;
					pthread_mutex_unlock(&num_edit_ops_mutex);
					hit->aligned = 1;
					return 0;
				}
				else return 0;
			}
				
			// too many mismatches?
			if (k1_aligned == -1) {
				return 0;
			}
	}
	
	// gapless alignment was successful
	hit->aligned = 1; 
	
	while (pthread_mutex_trylock(&num_edit_ops_mutex)) usleep(1);
	if (!thread_const->all_hit_strategy && hit->mismatches < NUM_EDIT_OPS) 
		NUM_EDIT_OPS = hit->mismatches;
	pthread_mutex_unlock(&num_edit_ops_mutex);	
	
	// successful alignment:
	return 0;
}


// returns 1 if alignment is gapless, 0 if there are gaps and -1 if nr of allowed MMs is exceeded 
int kbound_overhang_alignment(HIT* hit, THREAD_CONST* thread_const, int offset, int readstart, int num_thread)
{
	
	int K = thread_const->num_gaps;

	int length;
	int chrstart;
	char offset_comp = 0;
	
	if (readstart == 0) {
		length = hit->readpos + offset - 1;
		offset_comp = offset;
		if (hit->orientation == '+') chrstart = hit->start - length - 1;
			else chrstart = hit->end + length - 1;
	}
	else {
		length = thread_const->read_length - (hit->end - hit->start) - hit->readpos + offset;
		if (hit->orientation == '+') chrstart = hit->end;
			else chrstart = hit->start - 2;
	}

	int i,j,h;

	// Initialization:
	unsigned char score_before = hit->mismatches * thread_const->mm_score;
	if (readstart == 0) {
		for (i=0; i!=2*K+1; ++i) {
			M[num_thread][i][offset] = score_before;
			//T[num_thread][i][offset] = '0';
		}
		j = (K-offset < 0)? 0: K-offset;
		for (i=0; i!=j; ++i) {
			M[num_thread][0][i+1+offset] = score_before + (i+1) * thread_const->gap_score;
			//T[num_thread][0][i+1+offset] = UP;
		}
	}
	else {
		for (i = 0; i <= K; ++i) {
			M[num_thread][i][0] = score_before + i * thread_const->gap_score;
			if (i!=0) { 
				M[num_thread][0][i] = score_before + i * thread_const->gap_score;
				
				//T[num_thread][i][0] = LEFT;
				//T[num_thread][0][i] = UP;
			}
		}
		//T[num_thread][0][0] = '0';
	}
	
	// Alignment:
	int c, min_i = -1;
	char best_score_on_diag = 0;
	unsigned char score;
	unsigned char best_score = thread_const->worst_score + 1;
	unsigned char column_score = best_score;
	
	for (i = 1; i <= length; ++i) {
		
		for (h = -K; h <= K; ++h) {
			j = i + h;
		
				if (j <= length && j >= 1) {
					if (j>K) c = j - K;
						else c = 0;
					
				if ( !(readstart == 0 && j <= offset) && !(readstart != 0 && j > length-offset) ) {
					
					
					// Score-Function:
					if (hit->orientation == '+') {
						if (CHR_SEQ[hit->chromosome][chrstart+i-1] != READ[readstart+j-offset_comp-1] || !unique_base(READ[readstart+j-offset_comp-1]))
							 score = thread_const->mm_score;
						else score = thread_const->m_score;
					}
					else {
						if (get_compl_base(CHR_SEQ[hit->chromosome][chrstart-i+1]) != READ[readstart+j-offset_comp-1] || !unique_base(READ[readstart+j-offset_comp-1]))
							 score = thread_const->mm_score;
						else score = thread_const->m_score;
					}
					
					M[num_thread][i-c][j] = M[num_thread][i-c-(j<=K)][j-1] + score;
					//T[num_thread][i-c][j] = DIAGONAL;	// traceback diagonally
					
					if ((i-j-1 <= K) && (i-j-1 >= -K)) {
						if (M[num_thread][i-c-1][j] + thread_const->gap_score <= M[num_thread][i-c][j]) {
							
							if (i == j) {
								return 0;
							}
							
							M[num_thread][i-c][j] = M[num_thread][i-c-1][j] + thread_const->gap_score;
							//T[num_thread][i-c][j] = LEFT;	// traceback to the left
							
						}
					}
					if ((i-j+1 <= K) && (i-j+1 >= -K)) {
						if (M[num_thread][i-c+(j>K)][j-1] + thread_const->gap_score <= M[num_thread][i-c][j]) {
							
							if (i == j) {
								return 0;
							}
							
							M[num_thread][i-c][j] = M[num_thread][i-c+(j>K)][j-1] + thread_const->gap_score;
							//T[num_thread][i-c][j] = UP;	// traceback to upper
							
						}
					}
					
					if (readstart != 0 && j == length - offset) {
						if ( (!best_score_on_diag && M[num_thread][i-c][j] < best_score) || (best_score_on_diag && M[num_thread][i-c][j] <= best_score) ) {	
							// ensures that if score of most right bottom cell is equal to another cell in the same row, this other cell is reported with the best score
							best_score = M[num_thread][i-c][j];
							min_i = i;
							if (i == j) best_score_on_diag = 1;
						}
					}
					
					if (M[num_thread][i-c][j] < column_score) column_score = M[num_thread][i-c][j];
					
				}
			}
			
		} //for h
		
		if (column_score > thread_const->worst_score) {
			
			// readstart==0: score in most right bottom cell cannot become less than worst score
			// best_score > WORST_SCORE: there is no cell in the bottom row which has a valid alignment score 
			if (readstart == 0 || best_score > thread_const->worst_score) {
				return -1;
			} 
			
			break;
			
		} else {
			column_score = thread_const->worst_score + 1;
		}
		
	} //for i
	
	
	if (readstart == 0) {
		j = length;
		i = K - (j<K) * (K - j);
		best_score = M[num_thread][i][j];
		min_i = j;
	}
	else {
		j = length-offset;
		i = min_i - (j>K) * (j - K);
	}
	
	if (best_score > thread_const->worst_score) {
		return -1;
	}
	
	// if best score is worse than max nr of allowed mismatches...
	if (best_score > thread_const->worst_mm_score) {
		// ...and best score is in the most right bottom cell, there is no better path with gaps -> so we have too many MMs
		if (min_i == j) {
			return -1;
		}
		// ..and there is another cell with this score other than the most right bottom cell -> there is a path with a valid score, BUT with gaps! 
		else {
			return 0;
		}
	}
	
	// FOR READSTART != 0 ONLY: if best path doesn't start from the most right bottom corner of the matrix -> gaps are in best alignment, return 0
	//if (min_i != j && M[num_thread][i][j] != M[num_thread][K-(j<K)*(K-j)][j]) {	@TODO!!!!!!!
	if (min_i != j) {	// even if score in most right bottom cell is equal to best score -> path with gaps needs less edit ops -> perform global algnm.
		return 0;
	}
	// FOR READSTART == 0: traceback has to start in most right bottom corner

	

	// Traceback (if there had been gaps the procedure returned 0, so only mismatches are left -> only diagonal traceback)
	int readpos;
	if (readstart == 0) readpos = j - offset;
		else readpos = readstart + j;
	i = K - (j<K) * (K - j);

	int mms = (int) (M[num_thread][i][j] / thread_const->mm_score) - (int) (score_before / thread_const->mm_score);

	while (j != (readstart==0)*offset && mms != 0) {
		if (M[num_thread][i][j] != M[num_thread][i-(j<=K)][j-1]) {
			// create mismatch:
			
			if (hit->orientation == '+') hit->edit_op[hit->mismatches].pos = readpos;
			else hit->edit_op[hit->mismatches].pos = thread_const->read_length - readpos + 1;
			hit->edit_op[hit->mismatches].mm = 1;
			hit->mismatches++;
					
			--mms;
		}
		--readpos;
		--j;
		if (j<K) --i;
	}

	// successfully aligned

	return 1;
}


// k-bound global alignment algorithm:
int kbound_global_alignment(HIT* hit, THREAD_CONST* thread_const, int num_thread)
{
	int K = thread_const->num_gaps ;
	
	// delete possible mismatches found in k1_alignment or at the beg/end:
	hit->mismatches = 0;

	int chrstart, chrend, offset_front, offset_end;
	if (hit->orientation == '+') {
		chrstart = hit->start - hit->readpos;		// 0-initialized
		if (chrstart < K) offset_front = chrstart;
			else offset_front = K;
		
		chrend = chrstart + thread_const->read_length;			// 1-initialized
		if (chrend + K > CHR_LENGTH[hit->chromosome]) offset_end = CHR_LENGTH[hit->chromosome] - chrend;
			else offset_end = K;
		
		chrstart -= offset_front;
	}
	else {
		chrstart = hit->end + hit->readpos - 2;		// 0-initialized
		if (chrstart + K >= CHR_LENGTH[hit->chromosome]) offset_front = CHR_LENGTH[hit->chromosome] - chrstart - 1;
			else offset_front = K;
		
		chrend = chrstart - thread_const->read_length + 1;		// 0-initialized
		if (chrend - K < 0) offset_end = chrend;
			else offset_end = K;
			
		chrstart += offset_front;					// 0-initialized
	}

	int length = thread_const->read_length + offset_front + offset_end;
	
	int i,j,h;

	// Initialization:
	for (i=0; i!=2*K+1-(K-offset_front); ++i) {
		
		M[num_thread][i][offset_front] = 0;
		T[num_thread][i][offset_front] = '0';
		
		j = (K-offset_front < 0)? 0: K-offset_front;
		for (h=0; h!=j; ++h) {
			M[num_thread][0][h+1+offset_front] = (h+1) * thread_const->gap_score;
			T[num_thread][0][h+1+offset_front] = UP;
		}
	}
	
	
	// Alignment:
	int c;
	unsigned char best_score = thread_const->worst_score + 1;
	unsigned char column_score = best_score;
	unsigned score;
	int min_i = 0;	// start of traceback
	char best_score_on_diag = 0;
	
	for (i = 1; i <= length; ++i) {
		
		for (h = -K; h <= K; ++h) {
			j = i + h;
		
				if (j <= length && j >= 1) {
					if (j>K) c = j - K;
						else c = 0;
					
				if ( j > offset_front && j <= length-offset_end ) {
					
					
					// Score-Function:
					if (hit->orientation == '+') {
						if (CHR_SEQ[hit->chromosome][chrstart+i-1] != READ[j-offset_front-1] || !unique_base(READ[j-offset_front-1]))	//COMP
							 score = thread_const->mm_score;
						else score = thread_const->m_score;
					}
					else {
						if (get_compl_base(CHR_SEQ[hit->chromosome][chrstart-i+1]) != READ[j-offset_front-1] || !unique_base(READ[j-offset_front-1])) //COMP
							 score = thread_const->mm_score;
						else score = thread_const->m_score;
					}
						
					M[num_thread][i-c][j] = M[num_thread][i-c-(j<=K)][j-1] + score;
					T[num_thread][i-c][j] = DIAGONAL;	// traceback diagonally
					
					if ((i-j+1 <= K) && (i-j+1 >= -K)) {
						// gap in chr
						if (M[num_thread][i-c+(j>K)][j-1] + thread_const->gap_score == M[num_thread][i-c][j] && 
								((hit->orientation == '+' && thread_const->gaps_most_right) || (hit->orientation == '-' && !thread_const->gaps_most_right))) {
							M[num_thread][i-c][j] = M[num_thread][i-c+(j>K)][j-1] + thread_const->gap_score;
							T[num_thread][i-c][j] = UP;	// traceback to upper with gaps most right
						}
						else if (M[num_thread][i-c+(j>K)][j-1] + GAP_SCORE < M[num_thread][i-c][j]) {
							M[num_thread][i-c][j] = M[num_thread][i-c+(j>K)][j-1] + thread_const->gap_score;
							T[num_thread][i-c][j] = UP;	// traceback to upper
						}
					}
					
					if ((i-j-1 <= K) && (i-j-1 >= -K)) {
						// gap in read
						if (M[num_thread][i-c-1][j] + thread_const->gap_score == M[num_thread][i-c][j] && 
								((hit->orientation == '+' && thread_const->gaps_most_right) || (hit->orientation == '-' && !thread_const->gaps_most_right))) {
							M[num_thread][i-c][j] = M[num_thread][i-c-1][j] + thread_const->gap_score;
							T[num_thread][i-c][j] = LEFT;	// traceback to the left with gaps most right
						}
						else if (M[num_thread][i-c-1][j] + thread_const->gap_score < M[num_thread][i-c][j]) {
							M[num_thread][i-c][j] = M[num_thread][i-c-1][j] + thread_const->gap_score;
							T[num_thread][i-c][j] = LEFT;	// traceback to the left
						}
					}

					// Remember best score, i.e. start of traceback
					if (j == length - offset_end) {
						
						// gaps in reads preferred to gaps in chr:
						if ( (i <= j && M[num_thread][i-c][j] <= best_score) || (i > j && ((best_score_on_diag && M[num_thread][i-c][j] <  best_score) 
																		   || (!best_score_on_diag && M[num_thread][i-c][j] <= best_score))) ) {
						
							best_score = M[num_thread][i-c][j];
							min_i = i;
							if (i == j) best_score_on_diag = 1;
						}
					}
					// best_score preference:
					//	1. diagonal
					//	2. i > j (gaps in reads)
					//	3. i < j (gaps in chr)
					
					if (M[num_thread][i-c][j] < column_score) column_score = M[num_thread][i-c][j];
						
				}
			}
			
		} //for h
		
		if (column_score > thread_const->worst_score) {
			if (best_score > thread_const->worst_score) {

				return 0;
			}
			else {
				break;
			}
		} else {
			column_score = thread_const->worst_score + 1;
		}
		
	} //for i
	
	
	if (best_score > thread_const->worst_score) {
        	return 0;
	}


	j = length - offset_end;
	i = min_i - (j>K)*(j-K);
	int chrpos;
	if (hit->orientation == '+') {
		chrpos = chrstart + min_i - 1;
		hit->end_offset += min_i - j;	// correcting the read pos on chr for later output
	}
	else {
		chrpos = chrstart - min_i + 1;
		hit->start_offset -= min_i - j;	// correcting the read pos on chr for later output
	}
	int readpos = thread_const->read_length - 1;
	
	
	
	
	// Traceback:
	while (T[num_thread][i][j] != '0' && M[num_thread][i][j] != 0) {
			
		switch (T[num_thread][i][j]) {		
			case DIAGONAL: {
				
				if (M[num_thread][i-(j<=K)][j-1] != M[num_thread][i][j]) {
					if ((hit->mismatches - hit->gaps) < thread_const->num_mismatches && hit->mismatches < NUM_EDIT_OPS) {
						if (hit->orientation == '+') 
							hit->edit_op[hit->mismatches].pos = readpos + 1;
						else hit->edit_op[hit->mismatches].pos = thread_const->read_length - readpos;
						hit->edit_op[hit->mismatches].mm = 1;
						hit->mismatches++; 
					}
					else {
						return 0;	// discard hit
					}
				}
				
				i = i-(j<=K);
				j--;
				if (hit->orientation == '+') chrpos--;
					else chrpos++;
				readpos--;
				break;
			}
			
			case LEFT: {
		
				if (hit->mismatches < NUM_EDIT_OPS && (!thread_const->stringent_gaplimit || hit->gaps < thread_const->num_gaps)) {
					if (hit->orientation == '+') hit->edit_op[hit->mismatches].pos = readpos + 2;
					else hit->edit_op[hit->mismatches].pos = thread_const->read_length - readpos;
					hit->edit_op[hit->mismatches].mm = 0;
					hit->mismatches++;
					hit->gaps++;
				}
				else {
					return 0;	// discard hit
				}
					
				i--;
				if (hit->orientation == '+') chrpos--;
					else chrpos++;
				break;
			}
			
			case UP: {
				
				if (hit->mismatches < NUM_EDIT_OPS && (!thread_const->stringent_gaplimit || hit->gaps < thread_const->num_gaps)) {
					if (hit->orientation == '+') hit->edit_op[hit->mismatches].pos = -readpos - 1;
					else hit->edit_op[hit->mismatches].pos = -thread_const->read_length + readpos; 
					hit->edit_op[hit->mismatches].mm = 0;
					hit->mismatches++;
					hit->gaps++;
				}
				else {
					return 0;	// discard hit
				}
				
				i = i+(j>K);
				j--;
				readpos--;
				break;
			}
		}
		
	}
	
	if (hit->orientation == '+') hit->start_offset += i + (j>K)*(j-K) - j;
	else hit->end_offset -= i + (j>K)*(j-K) - j;
	
	// successfully aligned
	return 1;
}


#endif

