// Authors: Korbinian Schneeberger, Stephan Ossowski and Joerg Hagmann
// Copyright (C) 2008 by Max-Planck Institute for Developmental Biology, Tuebingen, Germany

#include "genomemapper.h"

int init_defaults();
int init_options(int argc, char *argv[]);
int init_index_file();
int init_meta_index_file();
int init_constants();
int init_query_file();
int init_output_file();
int init_leftover_file();
int init_genome_file();
int init_hit_lists();
int init_operators();
int init_statistic_vars();	
int init_alignment_structures();

int init(int argc, char *argv[]) 
{
	init_defaults();
	init_options(argc, argv);
	init_index_file();
	init_meta_index_file();
	init_query_file();
	init_output_file();
	if (strlen(LEFTOVER_FILE_NAME) > 0) init_leftover_file();
	init_genome_file();
	init_alignment_structures();
	init_hit_lists();

	return 0;
}

int init_alignment_structures() 
{
	REDUNDANT = 0;
	
	/////////////////////////////
	////// if you change NUM_EDIT_OPS here, then put alloc_hits_by_editops in init_hit_lists_operator() after this function!!!
	/////////////////////////////
	
	if (NUM_MISMATCHES > NUM_EDIT_OPS) {
		fprintf(stderr, "\n!!! WARNING: Nr. of allowed mismatches is set to nr. of allowed edit operations!\n\n");
		NUM_MISMATCHES = NUM_EDIT_OPS;
	}
	if (NUM_GAPS > NUM_EDIT_OPS) {
		fprintf(stderr, "\n!!! WARNING: Nr. of allowed gaps is set to nr. of allowed edit operations!\n\n");
		NUM_GAPS = NUM_EDIT_OPS;
	}

	if (M_SCORE == MM_SCORE) {
		fprintf(stderr, "ERROR: Sorry, until now, program cannot handle equal mismatch- and match-scores! Please restart with different values!\n");
		exit(1);
	}
	
	if (VERBOSE) printf("allowed edit operations: %d / allowed mismatches: %d / allowed gaps: %d\n%s strategy\t%s overhang alignment\n------------------\n", 
		NUM_EDIT_OPS, NUM_MISMATCHES, NUM_GAPS,	ALL_HIT_STRATEGY? "all hit": "best hit", OVERHANG_ALIGNMENT? "with": "without");
	
	
	if (GAP_SCORE > MM_SCORE) WORST_SCORE = NUM_GAPS * GAP_SCORE + (NUM_EDIT_OPS - NUM_GAPS) * MM_SCORE;
		else				  WORST_SCORE = NUM_MISMATCHES * MM_SCORE + (NUM_EDIT_OPS - NUM_MISMATCHES) * GAP_SCORE;
	WORST_MM_SCORE = NUM_MISMATCHES * MM_SCORE;
		
	
	ALIGNSEQ = (char *) malloc ((MAX_READ_LENGTH + 3 * MAX_EDIT_OPS) * sizeof(char));
	
	M = (double**) malloc ((2 * NUM_GAPS + 1) * sizeof(double*));
	T = (char**) malloc ((2 * NUM_GAPS + 1) * sizeof(char*));
	
	int i;
	for (i=0; i!=2*NUM_GAPS+1; ++i) {
		*(M+i) = (double*) malloc ((MAX_READ_LENGTH + NUM_GAPS * 2) * sizeof(double));
		*(T+i) = (char*) malloc ((MAX_READ_LENGTH + NUM_GAPS * 2) * sizeof(char));
	}
	
	//only for debugging purposes:
	//chrseq = (char *) malloc (MAX_READ_LENGTH * sizeof(char));
	//chrseq[0] = '\0';
	
	return(0);
}

int init_statistic_vars()
{
	int i;
	PERFECT_READS = 0;
	PERFECT_HITS = 0;
	PERFECT_HITS_REV = 0;
	NUM_HITS = 0;
	for (i=0; i!=37; ++i) HITS_LEN[i] = 0;
	HITS_MM = (unsigned int *) malloc ((MAX_EDIT_OPS + 1) * sizeof(unsigned int));
	for (i=0; i!=MAX_EDIT_OPS+1; ++i) HITS_MM[i] = 0;
	NUM_ALIGNMENTS = 0;
	NUM_WHOLE_ALIGNMENTS = 0;
	ENDSTART_MAPPED[0] = 0;
	ENDSTART_MAPPED[1] = 0;
	NOT_ALIGNED[0] = 0;
	NOT_ALIGNED[1] = 0;
	GAPS_ENCOUNTERED[0] = 0;
	GAPS_ENCOUNTERED[1] = 0;
	GAPS_ENCOUNTERED[2] = 0;
	TOO_MANY_MMS[0] = 0;
	TOO_MANY_MMS[1] = 0;
	BREAK_TB_IN_GLOBAL_ALIGNMENT = 0;
	BREAK_GLOBAL_ALIGNMENT[0] = 0;
	BREAK_GLOBAL_ALIGNMENT[1] = 0;
	CELLS_GLOBAL = 0;
	CELLS_OVERHANG = 0;
	
	W = 0;
	
	/*HITS_READPOS = (unsigned int **) malloc ((MAX_READ_LENGTH - INDEX_DEPTH + 1) * sizeof(unsigned int**));
	for (i=0; i!=MAX_READ_LENGTH-INDEX_DEPTH+1; ++i) {
		*(HITS_READPOS+i) = (unsigned int *) malloc ((MAX_READ_LENGTH - INDEX_DEPTH - i + 1) * sizeof(unsigned int*));
		 for (j=0; j!=MAX_READ_LENGTH - INDEX_DEPTH - i + 1; ++j) {
		 	*(*(HITS_READPOS+i)+j) = 0;
		 }
	}*/

	return(0);
}

int init_from_meta_index() 
{
	if (DEBUG) printf("init_constants\n");
	init_constants();
	if (DEBUG) printf("alloc genome memory\n");
	alloc_genome_memory();
	if (DEBUG) printf("init_operators\n");
	init_operators();
	if (STATISTICS) {
		if (DEBUG) printf("init_statistic_vars\n");
		init_statistic_vars();
	}
	if (!HITLEN_LIMIT) HITLEN_LIMIT = INDEX_DEPTH;
	if (DEBUG) printf("done init_from_meta_index\n");

	return(0);
}

int init_defaults() 
{
	DEBUG = 0;
	DEBUGREAD = 0;

	ALL_HIT_STRATEGY = 0;	// default: best hit strategy
	HITLEN_LIMIT = 0;		// only temporarily 0
	
	int i;

	VERBOSE = 0;

	MAP_REVERSE = 1;

	POWER[0] = 1;
	for (i = 1 ; i <= MAX_INDEX_DEPTH; i++) {
		POWER[i] = POWER[i-1] * 4;
	}
	
	HAS_SLOT = 0;
	READS_MAPPED = 0;
	NUM_EDIT_OPS = 4;
	NUM_GAPS = 1;
	NUM_MISMATCHES = 3;
	M_SCORE = 0;
	MM_SCORE = 4;
	GAP_SCORE = 5;
	GAPS_MOST_RIGHT = 0;
	OVERHANG_ALIGNMENT = 1;
	STRINGENT_GAPLIMIT = 1;
	PRINT_SEQ = 0;
	FLANKING = 0;
	
	OUTPUT_FORMAT = 0;
	
	SCORES_OUT = 1;
	
	REPEATMAP = 0;
		
	CHROM_CONTAINER_SIZE = 15000000;
	
	STATISTICS = 0;
	
	READ_ID = (char *) malloc(MAX_READ_ID_LENGTH * sizeof(char));
	READ = (char *) malloc(MAX_READ_LENGTH * sizeof(char));
	for (i = 0; i != 3; ++i) READ_QUALITY[i] = (char*) malloc(MAX_READ_LENGTH * sizeof(char));
	linenr = 0;
	READ_LENGTH = 0;
	
	listcount = 0;
	listocc = 0;
	
	return(0);
}

int init_options(int argc, char *argv[]) 
{
	int i;
	char not_defined;
	char has_index = 0;
	char has_meta_index = 0;
	char has_query = 0;
	char has_genome = 0;
	char output[5];

	for (i = 1; i < argc; i++) {
		not_defined = 1;

		//genome file
		if(strcmp(argv[i],"-i")==0){
			not_defined = 0;
			if(i+1 > argc - 1){ usage(); exit(1); }
			i++;
			strcpy(GENOME_FILE_NAME, argv[i]);
			has_genome = 1;
		}
		
		//index file
		if(strcmp(argv[i],"-x")==0){
			not_defined = 0;
			if(i+1 > argc - 1) { usage(); exit(1); }
			i++;
			strcpy(INDEX_FILE_NAME, argv[i]);
			has_index = 1;
		}

		//meta index file
		if(strcmp(argv[i],"-t")==0){
			not_defined = 0;
			if(i+1 > argc - 1) { usage(); exit(1); }
			i++;
			strcpy(META_INDEX_FILE_NAME, argv[i]);
			has_meta_index = 1;
		}

		//query file
		if(strcmp(argv[i],"-q")==0){
			not_defined = 0;
			if(i+1 > argc - 1){ usage(); exit(1); }
			i++;
			strcpy(QUERY_FILE_NAME, argv[i]);
			has_query = 1;
		}

    	//output file
		if(strcmp(argv[i],"-o")==0){
			not_defined = 0;
			if(i+1 > argc - 1){ usage(); exit(1); }
			i++;
			strcpy(OUT_FILE_NAME, argv[i]);
		}
		
		//output format
		if(strcmp(argv[i],"-f")==0){
			not_defined = 0;
			if(i+1 > argc - 1){ usage(); exit(1); }
			i++;
			printf("argv .%s.\n",argv[i]);
			strcpy(output, argv[i]);
			if (strcmp(output, "shore") == 0 || strcmp(output, "SHORE") == 0) {
				OUTPUT_FORMAT = 0;
			}
			else if (strcmp(output, "bed") == 0 || strcmp(output, "BED") == 0) {
				OUTPUT_FORMAT = 1;
			}
			else {
				fprintf(stderr, "ERROR: Output file format must either be \"shore\" or \"bed\"\n");
				exit(0);
			}
		}
		
		//leftover file
		if(strcmp(argv[i],"-u")==0){
			not_defined = 0;
			if(i+1 > argc - 1){ usage(); exit(1); }
			i++;
			strcpy(LEFTOVER_FILE_NAME, argv[i]);
		}

		//verbose
		if(strcmp(argv[i],"-v")==0){
			not_defined = 0;
			VERBOSE = 1;
		}
		
		//scores out
		if(strcmp(argv[i],"-e")==0){
			not_defined = 0;
			SCORES_OUT = 0;
		}

		//build reverse index
		if(strcmp(argv[i],"-r")==0){
			not_defined = 0;
			MAP_REVERSE = 0;
		}
		
		//report all hits-strategy
		if(strcmp(argv[i],"-a")==0){
			not_defined = 0;
			ALL_HIT_STRATEGY = 1;
		}
		// else best hit strategy
		
		//max nr of alignments per read
		if(strcmp(argv[i],"-ar")==0){
			not_defined = 0;
			if (i+1 > argc - 1){ usage(); exit(1); }
			i++;
			if (REPEATMAP != 0) {
				fprintf(stderr, "ERROR: options -a and -ar exclude themselves!\n");
				exit(0);
			}
			if ((REPEATMAP = atoi(argv[i])) == 0) {
				if (argv[i][0] != '0') {
					fprintf(stderr, "ERROR: Number of alignments must be an integer value!\n");
					exit(0);
				}
			}
		}
		
		//max nr of alignments per read, randomly chosen!
		if(strcmp(argv[i],"-n")==0){
			not_defined = 0;
			if (i+1 > argc - 1){ usage(); exit(1); }
			i++;
			if (REPEATMAP != 0) {
				fprintf(stderr, "ERROR: options -a and -ar exclude themselves!\n");
				exit(0);
			}
			if ((REPEATMAP = atoi(argv[i])) == 0) {
				if (argv[i][0] != '0') {
					fprintf(stderr, "ERROR: Number of alignments must be an integer value!\n");
					exit(0);
				}
			}
			REPEATMAP = -REPEATMAP;
		}
		
		//max number of allowed edit operations
		if(strcmp(argv[i],"-E")==0){
			not_defined = 0;
			if (i+1 > argc - 1){ usage(); exit(1); }
			i++;
			if ((NUM_EDIT_OPS = atoi(argv[i])) == 0) {
				if (argv[i][0] != '0') {
					fprintf(stderr, "ERROR: Number of edit operations must be an integer value!\n");
					exit(0);
				}
			}
			if (NUM_EDIT_OPS > MAX_EDIT_OPS) {
				fprintf(stderr, "ERROR: Number of allowed mismatches exceeds maximal number of edit operations (=%d)! Please restart with a lower value!\n", MAX_EDIT_OPS);
				exit(0);
			}
		}
		
		//max number of allowed mismatches
		if(strcmp(argv[i],"-M")==0){
			not_defined = 0;
			if (i+1 > argc - 1){ usage(); exit(1); }
			i++;
			if ((NUM_MISMATCHES = atoi(argv[i])) == 0) {
				 if (argv[i][0] != '0') {
					fprintf(stderr, "ERROR: Number of mismatches must be an integer value!\n");
					exit(0);
				}
			}
			if (NUM_MISMATCHES > MAX_EDIT_OPS) {
				fprintf(stderr, "ERROR: Number of allowed mismatches exceeds maximal number of edit operations (=%d)! Please restart with a lower value!\n", MAX_EDIT_OPS);
				exit(0);
			}
		}
		
		//max number of allowed gaps
		if(strcmp(argv[i],"-G")==0){
			not_defined = 0;
			if (i+1 > argc - 1){ usage(); exit(1); }
			i++;
			if ((NUM_GAPS = atoi(argv[i])) == 0) {
				if (argv[i][0] != '0') {
					fprintf(stderr, "ERROR: Number of gaps must be an integer value!\n");
					exit(0);
				}
			}
			if (NUM_GAPS > MAX_EDIT_OPS) {
				fprintf(stderr, "ERROR: Number of allowed gaps exceeds maximal number of edit operations (=%d)! Please restart with a lower value!\n", MAX_EDIT_OPS);
				exit(0);
			}
		}
	
		//match score
		if(strcmp(argv[i],"-match_score")==0){
			not_defined = 0;
			if (i+1 > argc - 1){ usage(); exit(1); }
			i++;
			M_SCORE = atof(argv[i]);
		}
		
		//mismatch score
		if(strcmp(argv[i],"-m")==0){
			not_defined = 0;
			if (i+1 > argc - 1){ usage(); exit(1); }
			i++;
			MM_SCORE = atof(argv[i]);
			if (MM_SCORE < 0) {
				fprintf(stderr, "ERROR: Mismatch score must be positive!\n");
				exit(0);
			}
			if (MM_SCORE == 0) fprintf(stderr, "\n!!! WARNING: Mismatch score is 0! This could lead to bad alignments!\n\n");
		}
		
		//gap score
		if(strcmp(argv[i],"-g")==0){
			not_defined = 0;
			if (i+1 > argc - 1){ usage(); exit(1); }
			i++;
			GAP_SCORE = atof(argv[i]);
			if (GAP_SCORE < 0) {
				fprintf(stderr, "ERROR: Gap score must be positive!\n");
				exit(0);
			}
			if (GAP_SCORE == 0) fprintf(stderr, "\n!!! WARNING: Gap score is 0! This could lead to bad alignments!\n\n");
		}
		
		//hitlength limit
		if(strcmp(argv[i],"-l")==0){
			not_defined = 0;
			if (i+1 > argc - 1){ usage(); exit(1); }
			i++;
			if ((HITLEN_LIMIT = atoi(argv[i])) == 0) {
				fprintf(stderr, "ERROR: Hitlength limit must be an integer value unequal to 0!\n");
				exit(0);
			}
		}
		
		//chromosome container size
		if(strcmp(argv[i],"-c")==0){
			not_defined = 0;
			if (i+1 > argc - 1){ usage(); exit(1); }
			i++;
			if ((CHROM_CONTAINER_SIZE += atoi(argv[i])) == 0) {
				fprintf(stderr, "ERROR: Chromosome Container Size must be an integer value unequal to 0!\n");
				exit(0);
			}
		}
		
		//nr of unallowed chars in reads
		/*if(strcmp(argv[i],"-n")==0){
			not_defined = 0;
			if (i+1 > argc - 1){ usage(); exit(1); }
			i++;
			if ((NUM_UNALLOWED_CHARS = atoi(argv[i])) == 0 && argv[i][0] != '0') {
				fprintf(stderr, "ERROR: Number of non-base symbols in read must be an integer value!\n");
				exit(0);
			}
		}*/
		
		//gaps most right
		if(strcmp(argv[i],"-d")==0){
			not_defined = 0;
			GAPS_MOST_RIGHT = 1;
		}
		
		//overhang alignment
		if(strcmp(argv[i],"-h")==0){
			not_defined = 0;
			OVERHANG_ALIGNMENT = 0;
		}
		
		//overhang alignment
		if(strcmp(argv[i],"-w")==0){
			not_defined = 0;
			STRINGENT_GAPLIMIT = 0;
		}
		
		//statistics
		if(strcmp(argv[i],"-s")==0){
			not_defined = 0;
			STATISTICS = 1;
		}
		
		//print out gene, too (for every hit) used for WMD2
		if(strcmp(argv[i],"-target_info")==0){
			not_defined = 0;
			if (i+1 > argc - 1){ fprintf(stderr, "-target_info needs an integer value (1: length only, 2: length+sequence)\n"); exit(0); }
			i++;
			if (((PRINT_SEQ = atoi(argv[i])) == 0) || PRINT_SEQ < 1 || PRINT_SEQ > 2) {
				fprintf(stderr, "-target_info value must be either 1 or 2!\n");
				exit(0);
			}
		}
		
		//flanking region of a hit
		if(strcmp(argv[i],"-flanking")==0){
			not_defined = 0;
			if (i+1 > argc - 1){ fprintf(stderr, "-flanking needs an integer value, the length of one of the symmetric flanking regions of a hit\n"); exit(0); }
			i++;
			if (((FLANKING = atoi(argv[i])) == 0) || FLANKING < 0 || FLANKING > 100) {
				fprintf(stderr, "-flanking value must be a positive integer and must not exceed 100!\n");
				exit(0);
			}
		}
		
		//debug
		if(strcmp(argv[i],"-p")==0){
			not_defined = 0;
			if (i+1 > argc-1) DEBUG = 1;
			else if ((DEBUGREAD = atoi(argv[i+1])) != 0) i++;
			else DEBUG = 1;
			printf("read %d will be debugged\n", DEBUGREAD);
		}

		if (not_defined == 1) {
			usage(); exit(1);
		}
	}

	if (has_index == 0 || has_query == 0 || has_meta_index == 0 || has_genome == 0) {
		usage(); exit(1);
	}
	
	if (REPEATMAP < 0 && ALL_HIT_STRATEGY) {
		fprintf(stderr, "ERROR: All hit strategy and randomly chosen output alignments exclude each other (options -A and -ar)!\n");
		exit(0);
	}

	return 0;
}

int init_constants() 
{
		if (INDEX_DEPTH == 5) {
				BINARY_CODE[0] = 0;			//binary number: 0000 0000 0000 0000 0000 0000
				BINARY_CODE[1] = 256;		//binary number: 0000 0000 0000 0001 0000 0000
				BINARY_CODE[2] = 512;		//binary number: 0000 0000 0000 0010 0000 0000
				BINARY_CODE[3] = 768;		//binary number: 0000 0000 0000 0011 0000 0000
		}
		if (INDEX_DEPTH == 6) {
				BINARY_CODE[0] = 0;			//binary number: 0000 0000 0000 0000 0000 0000
				BINARY_CODE[1] = 1024;		//binary number: 0000 0000 0000 0100 0000 0000
				BINARY_CODE[2] = 2048;		//binary number: 0000 0000 0000 1000 0000 0000
				BINARY_CODE[3] = 3072;		//binary number: 0000 0000 0000 1100 0000 0000
		}
		if (INDEX_DEPTH == 7) {
				BINARY_CODE[0] = 0;			//binary number: 0000 0000 0000 0000 0000 0000
				BINARY_CODE[1] = 4096;		//binary number: 0000 0000 0001 0000 0000 0000
				BINARY_CODE[2] = 8192;		//binary number: 0000 0000 0010 0000 0000 0000
				BINARY_CODE[3] = 12288;		//binary number: 0000 0000 0011 0000 0000 0000
		}
        if (INDEX_DEPTH == 8) {
                BINARY_CODE[0] = 0;             //binary number: 0000 0000 0000 0000 0000 0000
                BINARY_CODE[1] = 16384;         //binary number: 0000 0000 0100 0000 0000 0000
                BINARY_CODE[2] = 32768;         //binary number: 0000 0000 1000 0000 0000 0000
                BINARY_CODE[3] = 49152;         //binary number: 0000 0000 1100 0000 0000 0000
        }
        if (INDEX_DEPTH == 9) {
                BINARY_CODE[0] = 0;             //binary number: 0000 0000 0000 0000 0000 0000
                BINARY_CODE[1] = 65536;         //binary number: 0000 0001 0000 0000 0000 0000
                BINARY_CODE[2] = 131072;        //binary number: 0000 0010 0000 0000 0000 0000
                BINARY_CODE[3] = 196608;        //binary number: 0000 0011 0000 0000 0000 0000
        }
        if (INDEX_DEPTH == 10) {
                BINARY_CODE[0] = 0;             //binary number: 0000 0000 0000 0000 0000 0000
                BINARY_CODE[1] = 262144;        //binary number: 0000 0100 0000 0000 0000 0000
                BINARY_CODE[2] = 524288;        //binary number: 0000 1000 0000 0000 0000 0000
                BINARY_CODE[3] = 786432;        //binary number: 0000 1100 0000 0000 0000 0000
        }
        if (INDEX_DEPTH == 11) {
                BINARY_CODE[0] = 0;             //binary number: 0000 0000 0000 0000 0000 0000
                BINARY_CODE[1] = 1048576;       //binary number: 0001 0000 0000 0000 0000 0000
                BINARY_CODE[2] = 2097152;       //binary number: 0010 0000 0000 0000 0000 0000
                BINARY_CODE[3] = 3145728;       //binary number: 0011 0000 0000 0000 0000 0000
        }
        if (INDEX_DEPTH == 12) {
                BINARY_CODE[0] = 0;             //binary number: 0000 0000 0000 0000 0000 0000
                BINARY_CODE[1] = 4194304;       //binary number: 0100 0000 0000 0000 0000 0000
                BINARY_CODE[2] = 8388608;       //binary number: 1000 0000 0000 0000 0000 0000
                BINARY_CODE[3] = 12582912;      //binary number: 1100 0000 0000 0000 0000 0000
        }
        if (INDEX_DEPTH == 13) {
                BINARY_CODE[0] = 0;             //binary number: 0000 0000 0000 0000 0000 0000 0000
                BINARY_CODE[1] = 16777216;      //binary number: 0001 0000 0000 0000 0000 0000 0000
                BINARY_CODE[2] = 33554432;      //binary number: 0010 0000 0000 0000 0000 0000 0000
                BINARY_CODE[3] = 50331648;      //binary number: 0011 0000 0000 0000 0000 0000 0000
        }

	return(0);
}

int init_output_file() 
{
	if (strlen(OUT_FILE_NAME) > 0) {
		if ((OUT_FP = fopen(OUT_FILE_NAME, "w")) == NULL) {
			fprintf(stderr, "ERROR : Couldn't open output file %s\n", OUT_FILE_NAME);
			exit(1);
		}
	} else {
		OUT_FP = stdout;
	}

	return(0);
}

int init_leftover_file() 
{
	if ((LEFTOVER_FP = fopen(LEFTOVER_FILE_NAME, "w")) == NULL) {
		fprintf(stderr, "ERROR : Couldn't open leftover file %s\n", LEFTOVER_FILE_NAME);
		exit(1);
	}

	return(0);
}

int init_query_file() 
{
	if ((QUERY_FP = fopen(QUERY_FILE_NAME, "r")) == NULL) {
		fprintf(stderr, "ERROR : Couldn't open input file %s\n", QUERY_FILE_NAME);
		exit(1);
	}

	return(0);
}

int init_index_file() 
{
	if ((INDEX_FP = fopen(INDEX_FILE_NAME, "r")) == NULL) {
		fprintf(stderr, "ERROR : Couldn't open input file %s\n", INDEX_FILE_NAME);
		exit(1);
	}

	return(0);
}

int init_meta_index_file() 
{
	if ((META_INDEX_FP = fopen(META_INDEX_FILE_NAME, "r")) == NULL) {
		fprintf(stderr, "ERROR : Couldn't open input file %s\n", META_INDEX_FILE_NAME);
		exit(1);
	}


	return(0);
}

int init_genome_file() 
{
	if ((GENOME_FP = fopen(GENOME_FILE_NAME, "r")) == NULL) {
		fprintf(stderr, "ERROR : Couldn't open genome file %s\n", GENOME_FILE_NAME);
		exit(1);
	}

	return 0;
}


int init_hit_lists() 
{
	alloc_hit_lists_operator();	
	alloc_hits_by_score();	// A T T E N T I O N ! ! !   needs correct NUM_EDIT_OPS which can be changed in init_alignment_structures() !!!

	return(0);
}

int init_operators()
{
	MAPPING_ENTRY_OPERATOR = alloc_mapping_entry_container();
	MAPPING_ENTRY_OPERATOR_FIRST = MAPPING_ENTRY_OPERATOR;

	HIT_OPERATOR = alloc_hit_container();
	HIT_OPERATOR_FIRST = HIT_OPERATOR;

	alloc_chromosome_entry_container();

	return(0);
}
