// Authors: Korbinian Schneeberger, Stephan Ossowski and Joerg Hagmann
// Copyright (C) 2008 by Max-Planck Institute for Developmental Biology, Tuebingen, Germany

#include "genomemapper.h"

int kbound_overhang_alignment(HIT* hit, int offset, int readseq_start);
int kbound_global_alignment(HIT* hit);


char unique_base(char c)
{
	return (c == 'A' || c == 'C' || c == 'G' || c == 'T');
}

int align_hit_simple(HIT* hit)
{
	//DEBUG = 0;
	
	int hitlength = hit->end - hit->start + 1;
	int afterhit_len = READ_LENGTH - hit->readpos - hitlength + 1;
	
	// checking if read fits on the start or end of chromosome:
	//@TODO small overlaps should be mapped
	if (hit->orientation == '+') {
		if (hit->start < hit->readpos) {
			//printf("Read cannot be mapped on plus strand at start of chromosome %d!\n", hit->chromosome+1);
			if (STATISTICS) ENDSTART_MAPPED[0]++;
			return 0;
		}
		if (hit->end + afterhit_len > CHR_LENGTH[hit->chromosome]) {
			//printf("Read cannot be mapped on plus strand at end of chromosome %d!\n", hit->chromosome+1);
			if (STATISTICS) ENDSTART_MAPPED[0]++;
			return 0;
		}
	}
	
	if (hit->orientation == '-') {
		if (afterhit_len >= hit->start) {
			//printf("Read cannot be mapped on minus strand at start of chromosome %d!\n", hit->chromosome+1);
			if (STATISTICS) ENDSTART_MAPPED[0]++;
			return 0;
		}
		if (hit->end + hit->readpos - 1 > CHR_LENGTH[hit->chromosome]) {
			//printf("Read cannot be mapped on minus strand at end of chromosome %d!\n", hit->chromosome+1);
			if (STATISTICS) ENDSTART_MAPPED[0]++;
			return 0;
		}
	}
	
	if (STATISTICS) NUM_ALIGNMENTS++;
	
	int j;
	//int val;
	
	int readstart;
	if (hit->orientation == '+') {
		readstart = hit->start - hit->readpos;	// 0-initialized
	}
	else {
		readstart = hit->start - afterhit_len - 1;	//changed		0-initialized
	}
	
	if (DEBUG) {
		printf("---\n");
		printhit(hit);
		printf("genome seq before hit: ");
	}
	
	// from read[0] to read[hit->readpos]
	for (j=0; j!=hit->readpos-1; ++j) {
				if (DEBUG){ if (hit->orientation == '+') {printf("%d,%c ",hit->start - hit->readpos + j,CHR_SEQ[hit->chromosome][hit->start - hit->readpos + j]);}
					else printf("%d,%c ",hit->end + hit->readpos -2 -j,get_compl_base(CHR_SEQ[hit->chromosome][hit->end + hit->readpos - 2 - j]));}
		
		if (	(hit->orientation == '+') 
				&& (	(CHR_SEQ[hit->chromosome][hit->start - hit->readpos + j] != READ[j]) 
				 	 || !(unique_base(READ[j]))		// [XX] should also be a mismatch!
					// if read[j]=X and chr_seq not, then first or-condition is automatically true -> genome sequence doesn't have to be checked
				   ) 
		   ) 
		{
			// create mismatch:
			if (hit->mismatches < NUM_EDIT_OPS) {
				(hit->edit_op[hit->mismatches]).pos = j+1;
				(hit->edit_op[hit->mismatches]).mm = 1;
			}
	
			hit->mismatches++;
		}
			
		if (	(hit->orientation == '-') 
			 	&& (    (get_compl_base(CHR_SEQ[hit->chromosome][hit->end + hit->readpos - 2 - j]) != READ[j]) 
					 || !(unique_base(READ[j]))
				   )
		   )
		{
			// create mismatch:
			if (hit->mismatches < NUM_EDIT_OPS) {
				hit->edit_op[hit->mismatches].pos = READ_LENGTH - j;
				hit->edit_op[hit->mismatches].mm = 1;
			}
	
			hit->mismatches++;
		}
		
		if (hit->mismatches > NUM_EDIT_OPS) {
			/*if (hit->orientation == '+') val = readstart;
				else val = readstart * 2;
			if (READSTART[val]) REDUNDANT++;
				else READSTART[val] = 1;*/
			return 0;
		}
	}
	
	// from read[hit->readpos + hitlength] to read[READ_LENGTH - 1]
	int i = 0;
	j = hit->readpos + hitlength - 1;
	
	if (DEBUG) printf("j = %d\n",j);
	if (DEBUG) printf("genome seq after hit:  ");
	
	while ((hit->mismatches <= NUM_EDIT_OPS) && (j < READ_LENGTH)) {

				if (DEBUG){ if (hit->orientation == '+') {printf("%d,%c ", hit->end + i, CHR_SEQ[hit->chromosome][hit->end + i]);}
				else printf("%d,%c ", hit->start-2-i, get_compl_base(CHR_SEQ[hit->chromosome][hit->start - 2 - i]));}
		
		if (	(hit->orientation == '+') 
			 	&& (    (CHR_SEQ[hit->chromosome][hit->end + i] != READ[j]) 
					 || !(unique_base(READ[j]))		// [XX] should also be a mismatch!
					// if read[j]=X and chr_seq not, then first or-condition is automatically true -> genome sequence doesn't have to be checked
				   ) 
		   )
		{
			// create mismatch:
			if (hit->mismatches < NUM_EDIT_OPS) {		
				(hit->edit_op[hit->mismatches]).pos = j+1;
				(hit->edit_op[hit->mismatches]).mm = 1;
			}
	
			hit->mismatches++;
		}

			
		if (	(hit->orientation == '-') 
			 	&& (    (get_compl_base(CHR_SEQ[hit->chromosome][hit->start - 2 - i]) != READ[j]) 
					 || !(unique_base(READ[j]))
				   )
		   )
		{
			// create mismatch:
			if (hit->mismatches < NUM_EDIT_OPS) {
				(hit->edit_op[hit->mismatches]).pos = READ_LENGTH - j;
				(hit->edit_op[hit->mismatches]).mm = 1;
			}
	
			hit->mismatches++;
		}
		
		if (DEBUG) printhit(hit);
		if (hit->mismatches > NUM_EDIT_OPS) {
			/*if (hit->orientation == '+') val = readstart;
				else val = readstart * 2;
			if (READSTART[val]) REDUNDANT++;
				else READSTART[val] = 1;*/
			return 0;
		}
		
		++i;
		++j;
	}
	
	if (DEBUG) printf("durch\n");
	if (DEBUG) printhit(hit);
	
	if (hit->mismatches <= NUM_MISMATCHES) {	// there can't be gaps
		
		// insert hit into HITS_BY_SCORE
		insert_into_scorelist(hit, 1);
		
		if (!ALL_HIT_STRATEGY && hit->mismatches < NUM_EDIT_OPS) NUM_EDIT_OPS = hit->mismatches;
	}
	
	/*if (hit->orientation == '+') val = readstart;
		else val = readstart * 2;
	if (READSTART[val]) REDUNDANT++;
		else READSTART[val] = 1;*/

	return 0;
}


// returns if aligned hit fulfills MM and gap-criterias, thus is printed out (alignments are called in this method)
int prepare_kbound_alignment(HIT* hit)
{
	
	int hitlength = hit->end - hit->start + 1;
	int afterhit_len = READ_LENGTH - hit->readpos - hitlength + 1;
	
	// checking if read fits on the start or end of chromosome:
	//@TODO small overlaps should be mapped
	if (hit->orientation == '+') {
		if (hit->start < hit->readpos) {
			if (DEBUG) printf("Read cannot be mapped on plus strand at start of chromosome %d!\n", hit->chromosome+1);
			if (STATISTICS) ENDSTART_MAPPED[0]++;
			return 0;
		}
		if (hit->end + afterhit_len > CHR_LENGTH[hit->chromosome]) {
			if (DEBUG) printf("Read cannot be mapped on plus strand at end of chromosome %d %d %d!\n", hit->chromosome+1, hit->end, afterhit_len);
			if (STATISTICS) ENDSTART_MAPPED[0]++;
			return 0;
		}
	}
	
	if (hit->orientation == '-') {
		if (afterhit_len >= hit->start) {
			if (DEBUG) printf("Read cannot be mapped on minus strand at start of chromosome %d!\n", hit->chromosome+1);
			if (STATISTICS) ENDSTART_MAPPED[0]++;
			return 0;
		}
		if (hit->end + hit->readpos - 1 > CHR_LENGTH[hit->chromosome]) {
			if (DEBUG) printf("Read cannot be mapped on minus strand at end of chromosome %d!\n", hit->chromosome+1);
			if (STATISTICS) ENDSTART_MAPPED[0]++;
			return 0;
		}
	}
	
	
	// just perform global alignment if gap heuristic/speedup was disabled:
	if (!OVERHANG_ALIGNMENT) {
		if (kbound_global_alignment(hit)) {
			if (!ALL_HIT_STRATEGY && hit->mismatches < NUM_EDIT_OPS) NUM_EDIT_OPS = hit->mismatches;
			return insert_into_scorelist(hit, 1);
		}
		else return 0;
	}
	
	
	// perform whole alignment pipeline:
	
	char k1_aligned;
	int offset;
	
	int readstart;	// start pos of read on genome
	if (hit->orientation == '+') {
		readstart = hit->start - hit->readpos;		//	0-initialized
	}
	else {
		readstart = hit->start - afterhit_len - 1;	//changed		0-initialized
	}
	int readend = readstart + READ_LENGTH;	// end pos of read on genome	1-initialized
	
	if (DEBUG) printf("read %s, readstart %d, readend %d:\n",READ_ID, readstart, readend);
	if (DEBUG) printhit(hit);
	
	if (hit->readpos != 1) {
		
			if (DEBUG) printf("### VORNE ###\n");
		
			if (hit->orientation == '+') {
				if (readstart - NUM_GAPS < 0) offset = readstart;
					else offset = NUM_GAPS;
			} 
			else {
				if (readend + NUM_GAPS > CHR_LENGTH[hit->chromosome]) offset = CHR_LENGTH[hit->chromosome] - readend;
					else offset = NUM_GAPS;
			}

			// perform alignment
			k1_aligned = kbound_overhang_alignment(hit, offset, 0);
		
			// there are gaps on best path in alignment -> perform whole global alignment
			if (k1_aligned == 0) {
				if (DEBUG) printf("gaps encountered in front alignment -> global alignment initiated!\n");
							
				if (kbound_global_alignment(hit)) {
					if (!ALL_HIT_STRATEGY && hit->mismatches < NUM_EDIT_OPS) NUM_EDIT_OPS = hit->mismatches;
					return insert_into_scorelist(hit, 1);
				}
				else return 0;
			}
			
			// too many mismatches in aligned read already:	
			if (k1_aligned == -1) {
				if (DEBUG) printf("front alignment encountered too many mismatches\n");
				return 0;
			}
			
	}
	
	if (hit->readpos + hitlength != READ_LENGTH + 1) {
		
			if (DEBUG) printf("### HINTEN ###\n");

			if (hit->orientation == '+') {
				if (readend + NUM_GAPS > CHR_LENGTH[hit->chromosome]) offset = CHR_LENGTH[hit->chromosome] - readend;
					else offset = NUM_GAPS;
			}
			else {
				if (readstart - NUM_GAPS < 0) offset = readstart;
					else offset = NUM_GAPS;
			}

			// perform alignment if at least one edit op can still be afforded:
			if (hit->mismatches < NUM_EDIT_OPS) k1_aligned = kbound_overhang_alignment(hit, offset, hit->readpos+hitlength-1);
				else {
					if (DEBUG) printf("before end alignment: already too many mismatches\n");
					return 0;
				}
			
			// there are gaps on best path in alignment -> perform whole global alignment
			if (k1_aligned == 0) {
				if (DEBUG) printf("gaps encountered in end alignment -> global alignment initiated!\n");
			
				if (kbound_global_alignment(hit)) {
					if (!ALL_HIT_STRATEGY && hit->mismatches < NUM_EDIT_OPS) NUM_EDIT_OPS = hit->mismatches;
					return insert_into_scorelist(hit, 1);
				}
				else return 0;
			}
				
			// too many mismatches?
			if (k1_aligned == -1) {
				if (DEBUG) printf("end alignment encountered too many mismatches\n");
				return 0;
			}
	}
	
	// gapless alignment was successful -> insert hit into HITS_BY_SCORE:
	insert_into_scorelist(hit, 1);
	
	if (!ALL_HIT_STRATEGY && hit->mismatches < NUM_EDIT_OPS) NUM_EDIT_OPS = hit->mismatches;
	
	//DEBUG = 0;
	
	// successful alignment:
	return 1;
}


// returns 1 if alignment is gapless, 0 if there are gaps and -1 if nr of allowed MMs is exceeded 
int kbound_overhang_alignment(HIT* hit, int offset, int readstart)
{
	if (STATISTICS) NUM_ALIGNMENTS++;
	
	int K = NUM_GAPS;

	int length;
	int chrstart;
	char offset_comp = 0;
	
	if (readstart == 0) {
		length = hit->readpos + offset - 1;
		offset_comp = offset;
		if (hit->orientation == '+') chrstart = hit->start - length - 1;
			else chrstart = hit->end + length - 1;
	}
	else {
		length = READ_LENGTH - (hit->end - hit->start) - hit->readpos + offset;
		if (hit->orientation == '+') chrstart = hit->end;
			else chrstart = hit->start - 2;
	}

	if (DEBUG) printf("offset = %d, length = %d, chrstart = %d, readstart = %d\n\n",offset,length,chrstart,readstart);
	
	int i,j,h;
	
	// Initialization:
	if (DEBUG) printf("Initialization\n");
	double score_before = hit->mismatches * MM_SCORE;
	if (readstart == 0) {
		for (i=0; i!=2*K+1; ++i) {
			M[i][offset] = score_before;
			T[i][offset] = '0';
		}
		j = (K-offset < 0)? 0: K-offset;
		for (i=0; i!=j; ++i) {
			M[0][i+1+offset] = score_before + (i+1) * GAP_SCORE;
			T[0][i+1+offset] = UP;
		}
	}
	else {
		for (i = 0; i <= K; ++i) {
			M[i][0] = score_before + i * GAP_SCORE;
			if (i!=0) { 
				M[0][i] = score_before + i * GAP_SCORE;
				
				T[i][0] = LEFT;
				T[0][i] = UP;
			}
		}
		T[0][0] = '0';
	}
	
	// Alignment:
	int c, min_i = -1;
	char best_score_on_diag = 0;
	double score;
	double best_score = WORST_SCORE + 1;
	double column_score = best_score;
	
	if (DEBUG) printf("Alignment\n");
	for (i = 1; i <= length; ++i) {
		
		for (h = -K; h <= K; ++h) {
			j = i + h;
		
				if (j <= length && j >= 1) {
					if (j>K) c = j - K;
						else c = 0;
					
				if ( !(readstart == 0 && j <= offset) && !(readstart != 0 && j > length-offset) ) {
					
					if (STATISTICS) ++CELLS_OVERHANG;
					
					// Score-Function:
					if (hit->orientation == '+') {
						if (CHR_SEQ[hit->chromosome][chrstart+i-1] != READ[readstart+j-offset_comp-1] || !unique_base(READ[readstart+j-offset_comp-1]))
							 score = MM_SCORE;
						else score = M_SCORE;
						if (DEBUG) printf("i%d j%d chr[%d]=%c, read[%d]=%c -> score: %.1f\n", i,j,chrstart+i-1, CHR_SEQ[hit->chromosome][chrstart+i-1], 
							readstart+j-offset_comp-1, READ[readstart+j-offset_comp-1], score);
					}
					else {
						if (get_compl_base(CHR_SEQ[hit->chromosome][chrstart-i+1]) != READ[readstart+j-offset_comp-1] || !unique_base(READ[readstart+j-offset_comp-1]))
							 score = MM_SCORE;
						else score = M_SCORE;
						if (DEBUG) printf("i%d j%d chr[%d]=%c, read[%d]=%c -> score: %.1f\n", i,j,chrstart-i+1, get_compl_base(CHR_SEQ[hit->chromosome][chrstart-i+1]), 
							readstart+j-offset_comp-1, READ[readstart+j-offset_comp-1], score);
					}
					
					M[i-c][j] = M[i-c-(j<=K)][j-1] + score;
					T[i-c][j] = DIAGONAL;	// traceback diagonally
					
					if ((i-j-1 <= K) && (i-j-1 >= -K)) {
						if (M[i-c-1][j] + GAP_SCORE <= M[i-c][j]) {
							
							if (i == j) {
								if (readstart == 0) {
									if (DEBUG) print_alignment_matrix(chrstart,readstart,length,offset,0,hit->chromosome,hit->orientation,K);}
								else if (DEBUG) print_alignment_matrix(chrstart,readstart,length,0,offset,hit->chromosome,hit->orientation,K);
								if (STATISTICS) GAPS_ENCOUNTERED[0]++;
								return 0;
							}
							
							M[i-c][j] = M[i-c-1][j] + GAP_SCORE;
							T[i-c][j] = LEFT;	// traceback to the left
							
						}
					}
					if ((i-j+1 <= K) && (i-j+1 >= -K)) {
						if (M[i-c+(j>K)][j-1] + GAP_SCORE <= M[i-c][j]) {
							
							if (i == j) {
								if (readstart == 0) {
									if (DEBUG) print_alignment_matrix(chrstart,readstart,length,offset,0,hit->chromosome,hit->orientation,K);}
								else if (DEBUG) print_alignment_matrix(chrstart,readstart,length,0,offset,hit->chromosome,hit->orientation,K);
								if (STATISTICS) GAPS_ENCOUNTERED[0]++;
								return 0;
							}
							
							M[i-c][j] = M[i-c+(j>K)][j-1] + GAP_SCORE;
							T[i-c][j] = UP;	// traceback to upper
							
						}
					}
					
					if (readstart != 0 && j == length - offset) {
						if ( (!best_score_on_diag && M[i-c][j] < best_score) || (best_score_on_diag && M[i-c][j] <= best_score) ) {	
							// ensures that if score of most right bottom cell is equal to another cell in the same row, this other cell is reported with the best score
							best_score = M[i-c][j];
							min_i = i;
							if (i == j) best_score_on_diag = 1;
						}
					}
					
					if (M[i-c][j] < column_score) column_score = M[i-c][j];
					
				}
			}
			
		} //for h
		
		if (column_score > WORST_SCORE) {
			if (DEBUG) printf("Ha! break in overhang alignment!\n");
			if (DEBUG) printf("column_score %.1f best_score %.1f worst score %.1f\n",column_score, best_score, WORST_SCORE);
			
			// readstart==0: score in most right bottom cell cannot become less than worst score
			// best_score > WORST_SCORE: there is no cell in the bottom row which has a valid alignment score 
			if (readstart == 0 || best_score > WORST_SCORE) {
						if (readstart == 0) { if (DEBUG) print_alignment_matrix(chrstart, readstart, length, offset, 0, hit->chromosome, hit->orientation,K); }
						else if (DEBUG) print_alignment_matrix(chrstart, readstart, length, 0, offset, hit->chromosome, hit->orientation,K);
				if (DEBUG) printf("Alignment has too many edit ops! Discard hit!\n");
				if (STATISTICS) TOO_MANY_MMS[0]++;
				return -1;
			} 
			// if best_score has been found before most right bottom cell was processed
			/*else if (i < length - offset) {	// readstart here is != 0
						if (DEBUG) print_alignment_matrix(chrstart, readstart, length, 0, offset, hit->chromosome, hit->orientation,K);
				if (DEBUG) printf("Best path in alignment contains gaps!\n");
				GAPS_ENCOUNTERED[1]++;
				return 0;
			}*/
			
			break;
			
		} else {
			column_score = WORST_SCORE + 1;
		}
		
	} //for i
	
	
	if (readstart == 0) { if (DEBUG) print_alignment_matrix(chrstart, readstart, length, offset, 0, hit->chromosome, hit->orientation,K); }
	else if (DEBUG) print_alignment_matrix(chrstart, readstart, length, 0, offset, hit->chromosome, hit->orientation,K);


	if (readstart == 0) {
		j = length;
		i = K - (j<K) * (K - j);
		best_score = M[i][j];
		min_i = j;
	}
	else {
		j = length-offset;
		i = min_i - (j>K) * (j - K);
	}
	
	if (DEBUG) printf("best score: %.1f @ i %d (%d) j %d M[i][j] %.1f %s\n",best_score, min_i, i, j, M[i][j], (readstart==0)?"front":"end");

	if (best_score > WORST_SCORE) {
		if (DEBUG) printf("best score is worse than worst score -> discard hit\n");
		if (STATISTICS) TOO_MANY_MMS[1]++;
		return -1;
	}
	
	// if best score is worse than max nr of allowed mismatches...
	if (best_score > WORST_MM_SCORE) {
		// ...and best score is in the most right bottom cell, there is no better path with gaps -> so we have too many MMs
		if (min_i == j) {
			if (DEBUG) printf("best score is worse than worst score for MMs (and there is no other optimal path) -> discard hit\n");
			if (STATISTICS) TOO_MANY_MMS[1]++;
			return -1;
		}
		// ..and there is another cell with this score other than the most right bottom cell -> there is a path with a valid score, BUT with gaps! 
		else {
			if (DEBUG) printf("best path contains gaps, with MMs only score would be too bad\n");
			if (STATISTICS) GAPS_ENCOUNTERED[2]++;
			return 0;
		}
	}
	
	// FOR READSTART != 0 ONLY: if best path doesn't start from the most right bottom corner of the matrix -> gaps are in best alignment, return 0
	//if (min_i != j && M[i][j] != M[K-(j<K)*(K-j)][j]) {	@TODO!!!!!!!
	if (min_i != j) {	// even if score in most right bottom cell is equal to best score -> path with gaps needs less edit ops -> perform global algnm.
		if (DEBUG) printf("best path contains gaps, best score not in most right bottom cell\n");
		if (STATISTICS) GAPS_ENCOUNTERED[2]++;
		return 0;
	}
	// FOR READSTART == 0: traceback has to start in most right bottom corner

	

	// Traceback (if there had been gaps the procedure returned 0, so only mismatches are left -> only diagonal traceback)
	int readpos;
	if (readstart == 0) readpos = j - offset;
		else readpos = readstart + j;
	i = K - (j<K) * (K - j);

	int mms = (int) (M[i][j] / MM_SCORE) - (int) (score_before / MM_SCORE);
	if (DEBUG) printf("mismatches on diagonal: %d cos mms before have been: %d\n",mms, (int) (score_before/MM_SCORE));

	if (DEBUG) printf("readpos %d, readstart %d, mms %d, i %d, j %d, break for j at %d\n",readpos, readstart,mms,i,j, (readstart==0)*offset);
	
	while (j != (readstart==0)*offset && mms != 0) {
		if (DEBUG) printf("j %d, readpos %d, mms %d - M(i,j) %f  M(i,j-1) %f (i-j==1) %f\n", j, readpos,mms,M[i][j], M[i-(j<=K)][j-1], M[i-(j<=K)][j-1]);
		if (M[i][j] != M[i-(j<=K)][j-1]) {
			// create mismatch:
			
			if (hit->orientation == '+') hit->edit_op[hit->mismatches].pos = readpos;
				else					 hit->edit_op[hit->mismatches].pos = READ_LENGTH - readpos + 1;
			hit->edit_op[hit->mismatches].mm = 1;
			hit->mismatches++;
					
			--mms;
		}
		--readpos;
		--j;
		if (j<K) --i;
	}

	//print_alignment(hit,0);
	if (DEBUG) printf("i%d j%d M%f T%c\n",i,j,M[i][j],T[i][j]);
	
	if (DEBUG) printhit(hit);
	
	// successfully aligned
	return 1;
}






// k-bound global alignment algorithm:
int kbound_global_alignment(HIT* hit)
{
	if (STATISTICS) NUM_WHOLE_ALIGNMENTS++;
	
	int K = NUM_GAPS;
	
	// delete possible mismatches found in k1_alignment or at the beg/end:
	hit->mismatches = 0;

	int chrstart, chrend, offset_front, offset_end;
	if (hit->orientation == '+') {
		chrstart = hit->start - hit->readpos;		// 0-initialized
		if (chrstart < K) offset_front = chrstart;
			else offset_front = K;
		
		chrend = chrstart + READ_LENGTH;			// 1-initialized
		if (chrend + K > CHR_LENGTH[hit->chromosome]) offset_end = CHR_LENGTH[hit->chromosome] - chrend;
			else offset_end = K;
		
		chrstart -= offset_front;
	}
	else {
		chrstart = hit->end + hit->readpos - 2;		// 0-initialized
		if (chrstart + K >= CHR_LENGTH[hit->chromosome]) offset_front = CHR_LENGTH[hit->chromosome] - chrstart - 1;
			else offset_front = K;
		
		chrend = chrstart - READ_LENGTH + 1;		// 0-initialized
		if (chrend - K < 0) offset_end = chrend;
			else offset_end = K;
			
		chrstart += offset_front;					// 0-initialized
	}

	int length = READ_LENGTH + offset_front + offset_end;
	
	if (DEBUG) printf("offset_front = %d, offset_end = %d, length = %d, chrstart = %d, chrend = %d\n\n",offset_front,offset_end,length,chrstart,chrend);



	int i,j,h;
	
	// Initialization:
	if (DEBUG) printf("Initialization\n");
	for (i=0; i!=2*K+1-(K-offset_front); ++i) {
		
		M[i][offset_front] = 0;
		T[i][offset_front] = '0';
		
		j = (K-offset_front < 0)? 0: K-offset_front;
		if (DEBUG) printf("i %d j %d\n",i,j);
		for (h=0; h!=j; ++h) {
			M[0][h+1+offset_front] = (h+1) * GAP_SCORE;
			T[0][h+1+offset_front] = UP;
		}
	}
	
	
	// Alignment:
	int c;
	double best_score = WORST_SCORE + 1;
	double column_score = best_score;
	double score;
	int min_i = 0;	// start of traceback
	char best_score_on_diag = 0;
	
	if (DEBUG) printf("Alignment\n");
	for (i = 1; i <= length; ++i) {
		
		for (h = -K; h <= K; ++h) {
			j = i + h;
		
				if (j <= length && j >= 1) {
					if (j>K) c = j - K;
						else c = 0;
					
				if ( j > offset_front && j <= length-offset_end ) {
					
					if (STATISTICS) ++CELLS_GLOBAL;
					
					// Score-Function:
					if (hit->orientation == '+') {
						if (CHR_SEQ[hit->chromosome][chrstart+i-1] != READ[j-offset_front-1] || !unique_base(READ[j-offset_front-1]))	//COMP
							 score = MM_SCORE;
						else score = M_SCORE;
						//if (DEBUG) printf("i%d j%d chr[%d]=%c, read[%d]=%c -> score: %.1f\n", i,j,chrstart+i-1, CHR_SEQ[hit->chromosome][chrstart+i-1],	j-offset_front-1, READ[j-offset_front-1], score);	//COMP
					}
					else {
						if (get_compl_base(CHR_SEQ[hit->chromosome][chrstart-i+1]) != READ[j-offset_front-1] || !unique_base(READ[j-offset_front-1])) //COMP
							 score = MM_SCORE;
						else score = M_SCORE;
						//if (DEBUG) printf("i%d j%d chr[%d]=%c, read[%d]=%c -> score: %.1f\n", i,j,chrstart-i+1, get_compl_base(CHR_SEQ[hit->chromosome][chrstart-i+1]),	j-offset_front-1, READ[j-offset_front-1], score);	//COMP
					}
					
					M[i-c][j] = M[i-c-(j<=K)][j-1] + score;
					T[i-c][j] = DIAGONAL;	// traceback diagonally
					
					if ((i-j+1 <= K) && (i-j+1 >= -K)) {
						// gap in chr
						if (M[i-c+(j>K)][j-1] + GAP_SCORE == M[i-c][j] && 
								((hit->orientation == '+' && GAPS_MOST_RIGHT) || (hit->orientation == '-' && !GAPS_MOST_RIGHT))) {
							M[i-c][j] = M[i-c+(j>K)][j-1] + GAP_SCORE;
							T[i-c][j] = UP;	// traceback to upper with gaps most right
						}
						else if (M[i-c+(j>K)][j-1] + GAP_SCORE < M[i-c][j]) {
							M[i-c][j] = M[i-c+(j>K)][j-1] + GAP_SCORE;
							T[i-c][j] = UP;	// traceback to upper
						}
					}
					
					if ((i-j-1 <= K) && (i-j-1 >= -K)) {
						// gap in read
						if (M[i-c-1][j] + GAP_SCORE == M[i-c][j] && 
								((hit->orientation == '+' && GAPS_MOST_RIGHT) || (hit->orientation == '-' && !GAPS_MOST_RIGHT))) {
							M[i-c][j] = M[i-c-1][j] + GAP_SCORE;
							T[i-c][j] = LEFT;	// traceback to the left with gaps most right
						}
						else if (M[i-c-1][j] + GAP_SCORE < M[i-c][j]) {
							M[i-c][j] = M[i-c-1][j] + GAP_SCORE;
							T[i-c][j] = LEFT;	// traceback to the left
						}
					}

					// Remember best score, i.e. start of traceback
					if (j == length - offset_end) {
						
						// gaps in reads preferred to gaps in chr:
						if ( (i <= j && M[i-c][j] <= best_score) || (i > j && ((best_score_on_diag && M[i-c][j] <  best_score) 
																		   || (!best_score_on_diag && M[i-c][j] <= best_score))) ) {
						
							best_score = M[i-c][j];
							min_i = i;
							if (i == j) best_score_on_diag = 1;
						}
					}
					// best_score preference:
					//	1. diagonal
					//	2. i > j (gaps in reads)
					//	3. i < j (gaps in chr)
					
					if (M[i-c][j] < column_score) column_score = M[i-c][j];
						
				}
			}
			
		} //for h
		
		if (column_score > WORST_SCORE) {
			if (DEBUG) printf("Ha! break in global alignment!\n");
			if (DEBUG) printf("column_score %.1f best_score %.1f worst score %.1f\n",column_score, best_score, WORST_SCORE);
			if (best_score > WORST_SCORE) {
					if (DEBUG) print_alignment_matrix(chrstart, 0, length, offset_front, offset_end, hit->chromosome, hit->orientation,K);
				if (DEBUG) printf("Alignment has too many edit ops! Discard hit! (@ i %d)\n",i);
				if (STATISTICS) BREAK_GLOBAL_ALIGNMENT[0]++;
				return 0;
			}
			else {
				if (STATISTICS) BREAK_GLOBAL_ALIGNMENT[1]++;
				break;
			}
		} else {
			column_score = WORST_SCORE + 1;
		}
		
	} //for i
	
	if (DEBUG) print_alignment_matrix(chrstart, 0, length, offset_front, offset_end, hit->chromosome, hit->orientation, K);
	
	if (best_score > WORST_SCORE) {
		if (DEBUG) printf("Alignment has too many edit ops! Discard hit after whole alignment!\n");
                if (STATISTICS) BREAK_GLOBAL_ALIGNMENT[0]++;
        	return 0;
	}


	j = length - offset_end;
	i = min_i - (j>K)*(j-K);
	int chrpos;
	if (hit->orientation == '+') {
		chrpos = chrstart + min_i - 1;
		hit->end_offset += min_i - j;	// correcting the read pos on chr for later output
	}
	else {
		chrpos = chrstart - min_i + 1;
		hit->start_offset -= min_i - j;	// correcting the read pos on chr for later output
	}
	int readpos = READ_LENGTH - 1;
	
	
	
	if (DEBUG) printf("offset_front = %d, offset_end = %d, length = %d, chrstart = %d, chrend = %d, min_i %d\n\n",offset_front,offset_end,length,chrstart,chrend, min_i);
	if (DEBUG) printf("best score: %.1f, i=%d (%d), j=%d (%d)\n",best_score,i,chrpos,j,readpos);

	
	// Traceback:
	
	if (DEBUG) printf("Traceback\n");
	while (T[i][j] != '0' && M[i][j] != 0) {
			
		switch (T[i][j]) {		
			case DIAGONAL: {
				if (hit->orientation=='+') {
					if (DEBUG) printf("CHRSEQ[%d] = %c, READSEQ[%d] = %c  %.1f %.1f\n",chrpos,CHR_SEQ[hit->chromosome][chrpos],readpos,READ[readpos],M[i-(j<=K)][j-1], M[i][j]);
				} else
					if (DEBUG) printf("CHRSEQ[%d] = %c, READSEQ[%d] = %c  %.1f %.1f\n",chrpos,get_compl_base(CHR_SEQ[hit->chromosome][chrpos]),readpos,READ[readpos],M[i-(j<=K)][j-1],M[i][j]);
				
				
				//if (CHR_SEQ[hit->chromosome][chrpos] != READ[readpos] || !unique_base(READ[readpos])) {
				//if (M[i-(j<=K)][j-1] == M[i][j] - MM_SCORE) {					// doesn't work for doubles!!
				//if (fabs(M[i-(j<=K)][j-1] - M[i][j] + MM_SCORE) > 1E-9) {		// works, but requires math.h!
				if (M[i-(j<=K)][j-1] != M[i][j]) {
				if (DEBUG) printhit(hit);
					if ((hit->mismatches-hit->gaps) < NUM_MISMATCHES && hit->mismatches < NUM_EDIT_OPS) {
						if (hit->orientation == '+') hit->edit_op[hit->mismatches].pos = readpos + 1;
							else					 hit->edit_op[hit->mismatches].pos = READ_LENGTH - readpos;
						hit->edit_op[hit->mismatches].mm = 1;
						hit->mismatches++; 
					}
					else {
						if (STATISTICS) BREAK_TB_IN_GLOBAL_ALIGNMENT++;
						if (DEBUG) printf("break, too many Edit Ops!\n");
						return 0;	// discard hit
					}
				}
				
				i = i-(j<=K);
				j--;
				if (hit->orientation == '+') chrpos--;
					else chrpos++;
				readpos--;
				if (DEBUG) printf("i=%d, j=%d\n", i, j);
				break;
			}
			
			case LEFT: {
				if (hit->orientation == '+') {
					if (DEBUG) printf("CHRSEQ[%d] = %c, READSEQ gap at %d\n",chrpos,CHR_SEQ[hit->chromosome][chrpos],readpos+2);
				} else
					if (DEBUG) printf("CHRSEQ[%d] = %c, READSEQ gap at %d\n",chrpos,get_compl_base(CHR_SEQ[hit->chromosome][chrpos]),readpos+2);
		
				if (hit->mismatches < NUM_EDIT_OPS && (!STRINGENT_GAPLIMIT || hit->gaps < NUM_GAPS)) {
					if (STATISTICS && hit->gaps >= NUM_GAPS) W++;
					if (hit->orientation == '+') hit->edit_op[hit->mismatches].pos = readpos + 2;
						else					 hit->edit_op[hit->mismatches].pos = READ_LENGTH - readpos;
					hit->edit_op[hit->mismatches].mm = 0;
					hit->mismatches++;
					hit->gaps++;
				}
				else {
					if (STATISTICS) BREAK_TB_IN_GLOBAL_ALIGNMENT++;
					if (DEBUG) printf("break, too many Edit Ops or Gaps!\n");
					return 0;	// discard hit
				}
					
				i--;
				if (hit->orientation == '+') chrpos--;
					else chrpos++;
				break;
			}
			
			case UP: {
				if (DEBUG) printf("CHRSEQ gap at %d, READSEQ[%d] = %c\n",readpos+1,readpos,READ[readpos]);
				
				if (hit->mismatches < NUM_EDIT_OPS && (!STRINGENT_GAPLIMIT || hit->gaps < NUM_GAPS)) {
					if (STATISTICS && hit->gaps >= NUM_GAPS) W++;
					if (hit->orientation == '+') hit->edit_op[hit->mismatches].pos = -readpos - 1;
						else					 hit->edit_op[hit->mismatches].pos = -READ_LENGTH + readpos; 
					hit->edit_op[hit->mismatches].mm = 0;
					hit->mismatches++;
					hit->gaps++;
				}
				else {
					if (STATISTICS) BREAK_TB_IN_GLOBAL_ALIGNMENT++;
					if (DEBUG) printf("break, too many Edit Ops or Gaps!\n");
					return 0;	// discard hit
				}
				
				i = i+(j>K);
				j--;
				readpos--;
				break;
			}
		}
		
	}
	
	if (hit->orientation == '+') hit->start_offset += i + (j>K)*(j-K) - j;
		else hit->end_offset -= i + (j>K)*(j-K) - j;
	
	if (DEBUG) printhit(hit);
	
	// successfully aligned
	return 1;
}

